/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.IllegalClassException;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.PartName;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.query.DocumentTypeQuery;
import org.ametys.cms.search.query.FilenameQuery;
import org.ametys.cms.search.query.FullTextQuery;
import org.ametys.cms.search.query.MatchAllQuery;
import org.ametys.cms.search.query.MimeTypeGroupQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.cms.search.solr.SearcherFactory;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.explorer.cmis.CMISRootResourcesCollection;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationMode;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationResult;
import org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.explorer.threads.jcr.JCRPost;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.tag.TagAwareAmetysObject;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.documents.onlyoffice.OnlyOfficeManager;
import org.ametys.plugins.workspaces.html.HTMLTransformer;
import org.ametys.plugins.workspaces.indexing.solr.SolrWorkspacesConstants;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.search.query.KeywordQuery;
import org.ametys.plugins.workspaces.search.query.ProjectQuery;
import org.ametys.plugins.workspaces.tags.ProjectTagProviderExtensionPoint;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.util.AmetysHomeHelper;
import org.ametys.web.WebConstants;

/**
 * DAO for resources of a project
 */
public class WorkspaceExplorerResourceDAO extends ExplorerResourcesDAO implements PluginAware
{
    /** Avalon Role */
    @SuppressWarnings("hiding")
    public static final String ROLE = WorkspaceExplorerResourceDAO.class.getName();
    
    /**
     * Enumeration for resource type
     */
    public static enum ResourceType
    {
        /** Folder */
        FOLDER,
        /** File */
        FILE
    }
    
    /**
     * Enumeration for resource type for office
     */
    public static enum OfficeType
    {
        /** Word */
        WORD,
        /** Excel */
        EXCEL,
        /** Power point */
        POWERPOINT
    }
    
    /** resource operation helper */
    protected AddOrUpdateResourceHelper _addOrUpdateResourceHelper;
    
    private ProjectManager _projectManager;
    private SearcherFactory _searcherFactory;
    private WorkspaceModuleExtensionPoint _moduleEP;
    private WorkspacesHelper _workspaceHelper;
    private OnlyOfficeManager _onlyOfficeManager;
    private HTMLTransformer _htmlTransformer;
    private SourceResolver _sourceResolver;
    private ProjectMemberManager _projectMemberManager;
    private UserManager _userManager;
    
    private String _pluginName;
    
    private ProjectTagProviderExtensionPoint _tagProviderExtensionPoint;

    private I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _htmlTransformer = (HTMLTransformer) manager.lookup(HTMLTransformer.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _addOrUpdateResourceHelper = (AddOrUpdateResourceHelper) manager.lookup(AddOrUpdateResourceHelper.ROLE);
        _searcherFactory = (SearcherFactory) manager.lookup(SearcherFactory.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _onlyOfficeManager = (OnlyOfficeManager) manager.lookup(OnlyOfficeManager.ROLE);
        _tagProviderExtensionPoint = (ProjectTagProviderExtensionPoint) manager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    /**
     * Add a folder
     * @param parentId Identifier of the parent collection. Can be null to add folder to root folder.
     * @param inputName The desired name
     * @param description The folder description
     * @return The created folder or an error if a folder with same name already exists.
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    @Callable
    public Map<String, Object> addFolder(String parentId, String inputName, String description) throws IllegalAccessException
    {
        return addFolder(parentId, inputName, description, false);
    }
    
    /**
     * Add a folder
     * @param parentId Identifier of the parent collection. Can be null to add folder to root folder.
     * @param inputName The desired name
     * @param description The folder description
     * @param renameIfExists True to rename if existing
     * @return The result map with id, parentId and name keys
     */
    @Callable
    public Map<String, Object> addFolder(String parentId, String inputName, String description, Boolean renameIfExists)
    {
        ResourceCollection document = _getRootIfNull(parentId);
        if (document == null)
        {
            throw new IllegalArgumentException("Unable to add folder: parent folder not found");
        }
        
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_ADD, document) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to add folder without convenient right [" + RIGHTS_COLLECTION_ADD + "]");
        }
        
        if (!(document instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, document.getClass());
        }
        
        List<String> errors = new LinkedList<>();
        ResourceCollection collection = addResourceCollection((ModifiableResourceCollection) document, inputName, renameIfExists, errors);
        
        Map<String, Object> result = new HashMap<>();
        if (!errors.isEmpty())
        {
            result.put("message", errors.get(0));
            result.put("error", true);
        }
        
        if (collection != null)
        {
            if (StringUtils.isNotBlank(description))
            {
                ((ModifiableResourceCollection) collection).setDescription(description);
            }
            ((ModifiableResourceCollection) collection).saveChanges();
            
            result.putAll(_extractFolderData(collection));
        }
        
        return result;
    }
    
    /**
     * Move objects to parent folder
     * @param objectIds the object ids to move
     * @param parentFolderId the parent folder id
     * @return the results map
     * @throws RepositoryException if a repository exception occurred
     */
    @Callable
    public Map<String, Object> moveObjects(List<String> objectIds, String parentFolderId) throws RepositoryException
    {        
        for (String id : objectIds)
        {
            AmetysObject object = _resolver.resolveById(id);
            String rightId = object instanceof JCRResourcesCollection ? RIGHTS_COLLECTION_EDIT : RIGHTS_RESOURCE_RENAME;
            if (_rightManager.hasRight(_currentUserProvider.getUser(), rightId, object) != RightResult.RIGHT_ALLOW)
            {
                throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to move file or folder without convenient right [" + rightId + "]");
            }
        }
        
        JCRResourcesCollection parentFolder = (JCRResourcesCollection) _resolver.resolveById(parentFolderId);
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_EDIT, parentFolder) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit folder without convenient right [" + RIGHTS_COLLECTION_EDIT + "]");
        }
        
        Map<String, Object> results = moveObject(objectIds, parentFolder);
        if (results.containsKey("moved-objects"))
        {
            List<String> movedFolders = new ArrayList<>();
            List<String> movedFiles = new ArrayList<>();
            
            @SuppressWarnings("unchecked")
            List<String> movedObjects = (List<String>) results.get("moved-objects");
            for (String objectId : movedObjects)
            {
                AmetysObject object = _resolver.resolveById(objectId);
                if (object instanceof JCRResourcesCollection)
                {
                    movedFolders.add(objectId);
                }
                else
                {
                    movedFiles.add(objectId);
                }
            }
            
            results.put("moved-folders", movedFolders);
            results.put("moved-files", movedFiles);
        }
        
        return results;
    }
    
    /**
     * Rename a folder
     * @param id Identifier of the folder to edit
     * @param name The new name
     * @return The result map with id and name keys
     */
    @Callable
    public Map<String, Object> renameFolder(String id, String name)
    {
        Map<String, Object> result = new HashMap<>();
        
        JCRResourcesCollection folder = (JCRResourcesCollection) _resolver.resolveById(id);
        
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_EDIT, folder) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit folder without convenient right [" + RIGHTS_COLLECTION_EDIT + "]");
        }
        
        List<String> errors = new LinkedList<>();
        JCRResourcesCollection newFolder = null;
        try
        {
            newFolder = (JCRResourcesCollection) renameObject(folder, name, errors);
            if (!errors.isEmpty())
            {
                result.put("success", false);
                result.put("message", errors.get(0));
            }
            else
            {
                newFolder.saveChanges();

                result.put("success", true);
                result.putAll(_extractFolderData(newFolder));
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("Repository exception during folder edition.", e);
            errors.add("repository");
        }
        
        return result;
    }
    
    /**
     * Edit a folder
     * @param id Identifier of the folder to edit
     * @param inputName The desired name
     * @param description The folder description
     * @return The result map with id and name keys
     */
    @Callable
    public Map<String, Object> editFolder(String id, String inputName, String description)
    {
        JCRResourcesCollection folder = (JCRResourcesCollection) _resolver.resolveById(id);
        
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_EDIT, folder) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit folder without convenient right [" + RIGHTS_COLLECTION_EDIT + "]");
        }
        
        List<String> errors = new LinkedList<>();
        JCRResourcesCollection newFolder = null;
        boolean success = true;
        try
        {
            newFolder = StringUtils.isBlank(inputName) || folder.getName().equals(inputName)
                ? folder
                : (JCRResourcesCollection) renameObject(folder, inputName, errors);
        }
        catch (RepositoryException e)
        {
            success = false;
            getLogger().error("Repository exception during folder edition.", e);
            errors.add("repository");
        }
        
        Map<String, Object> result = new HashMap<>();
        if (!errors.isEmpty())
        {
            String error = errors.get(0);
            
            // existing node is allowed, the description can still be changed.
            if (!"already-exist".equals(error))
            {
                success = false;
                result.put("message", error);
            }
            else
            {
                newFolder = folder;
            }
        }
        
        if (success && newFolder != null)
        {
            if (StringUtils.isNotBlank(description) && !StringUtils.equals(newFolder.getDescription(), description))
            {
                newFolder.setDescription(description);
            }

            newFolder.saveChanges();
            result.putAll(_extractFolderData(newFolder));
        }
        
        result.put("success", success);
        return result;
    }
    
    /**
     * Delete a folder
     * @param id Identifier of the folder to delete
     * @return The result map with the parent id key
     */
    @Callable
    public Map<String, Object> deleteFolder(String id)
    {
        RemovableAmetysObject folder = (RemovableAmetysObject) _resolver.resolveById(id);
        
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_DELETE, folder) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to delete folder without convenient right [" + RIGHTS_COLLECTION_DELETE + "]");
        }
        
        List<String> errors = new LinkedList<>();
        String parentId = deleteObject(folder, errors);
        
        Map<String, Object> result = new HashMap<>();
        if (!errors.isEmpty())
        {
            String error = errors.get(0);
            result.put("message", error);
        }
        
        // include parent id except if it is the root
        ResourceCollection rootCollection = _getRootFromRequest();
        boolean isRoot = rootCollection != null && rootCollection.getId().equals(parentId);
        if (StringUtils.isNotEmpty(parentId) && !isRoot)
        {
            result.put("parentId", parentId);
        }
        
        return result;
    }
    
    /**
     * Lock resources
     * @param ids The id of resources to lock
     * @return the result.
     */
    @Callable
    public Map<String, Object> lockResources(List<String> ids)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("locked-resources", new ArrayList<>());
        result.put("unlocked-resources", new ArrayList<>());
        
        for (String id : ids)
        {
            JCRResource resource = _resolver.resolveById(id);
            
            if (!resource.isLocked())
            {
                resource.lock();
                
                @SuppressWarnings("unchecked")
                List<String> lockedResources = (List<String>) result.get("locked-resources");
                lockedResources.add(id);
            }
            else if (!resource.getLockOwner().equals(_currentUserProvider.getUser()))
            {
                UserIdentity lockOwner = resource.getLockOwner();
                
                getLogger().error("Unable to lock resource of id '" + id + "': the resource is already locked by user " + UserIdentity.userIdentityToString(lockOwner));
                
                Map<String, Object> info = new HashMap<>();
                info.put("id", id);
                info.put("name", resource.getName());
                info.put("lockOwner", _userHelper.user2json(lockOwner));
                
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> unlockedResources = (List<Map<String, Object>>) result.get("unlocked-resources");
                unlockedResources.add(info);
            }
        }
        
        return result;
    }
    
    /**
     * Unlock resources
     * @param ids The id of resources to lock
     * @return the result.
     */
    @Callable
    public Map<String, Object> unlockResources(List<String> ids)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        boolean canUnlockAll = _rightManager.hasRight(currentUser, RIGHTS_RESOURCE_UNLOCK_ALL, "/cms") == RightResult.RIGHT_ALLOW;
        
        Map<String, Object> result = new HashMap<>();
        
        result.put("unlocked-resources", new ArrayList<>());
        result.put("still-locked-resources", new ArrayList<>());
        
        for (String id : ids)
        {
            JCRResource resource = _resolver.resolveById(id);
            
            if (resource.isLocked())
            {
                if (canUnlockAll || resource.getLockOwner().equals(currentUser))
                {
                    resource.unlock();
                    
                    @SuppressWarnings("unchecked")
                    List<String> unlockedResources = (List<String>) result.get("unlocked-resources");
                    unlockedResources.add(id);
                }
                else
                {
                    UserIdentity lockOwner = resource.getLockOwner();
                    
                    getLogger().error("Unable to unlock resource of id '" + id + "': the resource is locked by user " + UserIdentity.userIdentityToString(lockOwner));
                    
                    Map<String, Object> info = new HashMap<>();
                    info.put("id", id);
                    info.put("name", resource.getName());
                    info.put("lockOwner", _userHelper.user2json(lockOwner));
                    
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> stilllockedResources = (List<Map<String, Object>>) result.get("still-locked-resources");
                    stilllockedResources.add(info);
                }
            }
            else
            {
                @SuppressWarnings("unchecked")
                List<String> unlockedResources = (List<String>) result.get("unlocked-resources");
                unlockedResources.add(id);
            }
        }
        
        return result;
    }
    
    /**
     * Determines if a resource with given name already exists
     * @param parentId the id of parent collection. Can be null.
     * @param name the name of resource
     * @return true if a resource with same name exists
     */
    @Callable
    @Override
    public boolean resourceExists(String parentId, String name)
    {
        ResourceCollection folder = _getRootIfNull(parentId);
        return folder != null && resourceExists(folder, name);
    }
    
    /**
     * Rename a folder
     * @param id Identifier of the folder to edit
     * @param name The new name
     * @return The result map with id and name keys
     */
    @Callable
    public Map<String, Object> renameFile(String id, String name)
    {
        Map<String, Object> result = new HashMap<>();
        
        JCRResource file = (JCRResource) _resolver.resolveById(id);
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_RENAME, file) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to rename file without convenient right [" + RIGHTS_RESOURCE_RENAME + "]");
        }
        
        if (StringUtils.isNotEmpty(name) && !StringUtils.equals(file.getName(), name))
        {
            List<String> errors = new LinkedList<>();
            JCRResource renamedFile = null;
            try
            {
                renamedFile = renameResource(file, name, errors);
                if (!errors.isEmpty())
                {
                    result.put("success", false);
                    result.put("message", errors.get(0));
                }
                else
                {
                    renamedFile.saveChanges();

                    result.put("success", true);
                    result.putAll(_extractFileData(renamedFile));
                }
            }
            catch (RepositoryException e)
            {
                getLogger().error("Repository exception during file edition.", e);
                errors.add("repository");
            }
            
        }
        
        return result;
    }
    
    
    /**
     * Edit a file
     * @param id Identifier of the file to edit
     * @param inputName The desired name
     * @param description The file description
     * @param tags The file tags 
     * @return The result map with id and name keys
     */
    @Callable
    public Map<String, Object> editFile(String id, String inputName, String description, Collection<String> tags)
    {
        Map<String, Object> result = new HashMap<>();
        JCRResource file = (JCRResource) _resolver.resolveById(id);
        
        // Check lock on resource
        if (!checkLock(file))
        {
            getLogger().warn("User '{}' is trying to edit file '{}' but it is locked by another user", _currentUserProvider.getUser(), file.getName());
            result.put("message", "locked");
            return result;
        }
        
        List<String> errors = new LinkedList<>();
        
        // Rename
        JCRResource renamedFile = null;
        if (StringUtils.isNotEmpty(inputName) && !StringUtils.equals(file.getName(), inputName))
        {
            if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_RENAME, file) != RightResult.RIGHT_ALLOW)
            {
                throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to rename file without convenient right [" + RIGHTS_RESOURCE_RENAME + "]");
            }
            
            try
            {
                renamedFile = renameResource(file, inputName, errors);
                
                if (errors.isEmpty())
                {
                    file = renamedFile;
                }
            }
            catch (RepositoryException e)
            {
                getLogger().error("Repository exception during folder edition.", e);
                errors.add("repository-rename");
            }
        }
        
        if (!errors.isEmpty())
        {
            String error = errors.get(0);
            result.put("message", error);
            return result;
        }
        
        // edit description + tags
        List<String> fileTags = _sanitizeFileTags(tags);
        
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_EDIT_DC, file) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit dc for file without convenient right [" + RIGHTS_RESOURCE_EDIT_DC + "]");
        }
        
        Map<String, Object> editValues = new HashMap<>();
        editValues.put("dc_description", StringUtils.defaultIfEmpty(description, null));
        
        try
        {
            file.setKeywords(fileTags.toArray(new String[fileTags.size()]));
            setDCMetadata(file, editValues);
            
            // Add tags to the project
            _projectManager.addTags(fileTags);
            
            file.saveChanges();
        }
        catch (AmetysRepositoryException e)
        {
            getLogger().error("Repository exception during folder edition.", e);
            errors.add("repository-edit");
        }
        
        if (!errors.isEmpty())
        {
            String error = errors.get(0);
            result.put("message", error);
        }
        else
        {
            result.putAll(_extractFileData(file));
        }
        
        return result;
    }
    
    /**
     * Set tags to resource
     * @param resourceId the id of resources
     * @param tags the file tags to set
     * @return the file tags
     */
    @Callable
    public Map<String, Object> setTags(String resourceId, List<Object> tags)
    {
        JCRResource file = (JCRResource) _resolver.resolveById(resourceId);
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_EDIT_DC, file) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to tag file without convenient right [" + RIGHTS_RESOURCE_EDIT_DC + "]");
        }
        
        List<Map<String, Object>> createdTagsJson = _workspaceHelper.handleTags(file, tags);
        
        file.saveChanges();
        
        Map<String, Object> results = new HashMap<>();
        results.put("fileTags", _tags2json(file));
        results.put("newTags", createdTagsJson);
        
        return results;
    }
    
    /**
     * Copy file resources
     * @param ids The list of identifiers for the resources to copy
     * @param targetId The id of target to copy into. Can be null to copy into root folder.
     * @return The result map with a message key in case of an error or with the list of uncopied/copied resources
     * @throws RepositoryException If there is a repository error
     */
    @Callable
    public Map<String, Object> copyFiles(List<String> ids, String targetId) throws RepositoryException
    {
        ResourceCollection folder = _getRootIfNull(targetId);
        if (folder == null)
        {
            throw new IllegalArgumentException("Unable to copy files: parent folder not found");
        }
        
        if (!(folder instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, folder.getClass());
        }
        
        return copyResource(ids, (ModifiableResourceCollection) folder);
    }
    
    /**
     * Move documents (files or folders)
     * @param ids The list of identifiers for the objects to move
     * @param targetId The id of target to move into. Can be null to move documents to root folder.
     * @return The result map with a message key in case of an error or with the list of unmoved/moved objects
     * @throws RepositoryException If there is a repository error
     */
    @Callable
    public Map<String, Object> moveDocuments(List<String> ids, String targetId) throws RepositoryException
    {
        ResourceCollection folder = _getRootIfNull(targetId);
        if (folder == null)
        {
            throw new IllegalArgumentException("Unable to move documents: parent folder not found");
        }
        
        if (!(folder instanceof JCRTraversableAmetysObject))
        {
            throw new IllegalClassException(JCRTraversableAmetysObject.class, folder.getClass());
        }
        
        return moveObject(ids, (JCRTraversableAmetysObject) folder);
    }
    
    /**
     * Search for files in the document module
     * @param query The search query
     * @param lang The search language
     * @return A result map containing a <em>resources</em> entry which is a list of the files data
     * @throws Exception if an exception occurs
     */
    @Callable
    public Map<String, Object> searchFiles(String query, String lang) throws Exception
    {
        // Handle result map
        Map<String, Object> result = new HashMap<>();
        
        String escapedQuery = query.replace("\"", "\\\"");
        
        Query solrQuery;
        if (StringUtils.isEmpty(query))
        {
            solrQuery = new MatchAllQuery();
        }
        else
        {
            List<Query> queries = new ArrayList<>();
            queries.add(new FilenameQuery(query));
            queries.add(new StringQuery(SolrFieldNames.TITLE, Operator.LIKE, "*" + query + "*", null));
            queries.add(new FullTextQuery(escapedQuery, lang));
            queries.add(new KeywordQuery(escapedQuery.split(" ")));
            solrQuery = new OrQuery(queries);
        }
        
        AmetysObjectIterable<Resource> results = _searcherFactory.create()
                .withQuery(solrQuery)
                .addFilterQuery(new DocumentTypeQuery(SolrWorkspacesConstants.TYPE_PROJECT_RESOURCE))
                .addFilterQuery(new ProjectQuery(_getProjectFromRequest().getId()))
                .search();
        
        List<Map<String, Object>> resourceData = results.stream()
            .map(this::_extractFileData)
            .collect(Collectors.toList());
        
        result.put("resources", resourceData);
        
        return result;
    }
    
    /**
     * Search for files by their type
     * @param type The file type
     * @return A result map containing a <em>resources</em> entry which is a list of the files data
     * @throws Exception if an exception occurs
     */
    @Callable
    public Map<String, Object> searchFilesByType(String type) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        AmetysObjectIterable<Resource> results = _searcherFactory.create()
                .withQuery(new MimeTypeGroupQuery(type))
                .addFilterQuery(new DocumentTypeQuery(SolrWorkspacesConstants.TYPE_PROJECT_RESOURCE))
                .addFilterQuery(new ProjectQuery(_getProjectFromRequest().getId()))
                .search();
        
        List<Map<String, Object>> resourceData = results.stream()
            .map(this::_extractFileData)
            .collect(Collectors.toList());
        
        result.put("resources", resourceData);
        
        return result;
    }
    
    /**
     * Get the root folder
     * @return the root folder as JSON object
     */
    @Callable
    public Map<String, Object> getRootFolder()
    {
        return getFolder(null);
    }
    
    /**
     * Get folder
     * @param folderId the folder id
     * @return the folder as JSON object
     */
    @Callable
    public Map<String, Object> getFolder(String folderId)
    {
        ResourceCollection collection = _getRootIfNull(folderId);
        return _extractFolderData(collection);
    }
    
    /**
     * Get file
     * @param resourceId the resource id
     * @return the file as JSON object
     */
    @Callable
    public Map<String, Object> getFile(String resourceId)
    {
        Resource resource = _resolver.resolveById(resourceId);
        return _extractFileData(resource);
    }
    /**
     * Get the child folders
     * @param parentId the parent id. Can be null to get root folders
     * @return the sub folders
     */
    @Callable
    public List<Map<String, Object>> getFolders(String parentId)
    {
        return getChildDocumentsData(parentId, false, true);
    }
    
    /**
     * Get the child files
     * @param parentId the parent id. Can be null to get root files
     * @return the child files
     */
    @Callable
    public List<Map<String, Object>> getFiles(String parentId)
    {
        return getChildDocumentsData(parentId, true, false);
    }
    
    /**
     * Get the children folders and files of the folder owning the given file
     * @param fileId the file id
     * @return the children folders and files
     */
    @Callable
    public Map<String, Object> getFileParentInfo(String fileId)
    {
        try
        {
            Resource file = _resolver.resolveById(fileId);
            AmetysObject parent = file.getParent();
            String parentId = (parent != null && parent instanceof ResourceCollection) ? parent.getId() : null;
            
            return getFoldersAndFiles(parentId);
        }
        catch (AmetysRepositoryException e) 
        {
            return Map.of("error", "invalid-item-id");
        }
    }
    
    /**
     * Get the child folders and files
     * @param folderId the parent id. Can be null to get root folders and files
     * @return the child folders and files
     */
    @Callable
    public Map<String, Object> getFoldersAndFiles(String folderId)
    {
        ResourceCollection root = _getRootFromRequest();
        ResourceCollection collection = StringUtils.isNotEmpty(folderId)
            ? (ResourceCollection) _resolver.resolveById(folderId)
            : root;
        
        Map<String, Object> data = _extractFolderData(collection);
        data.put("root", collection.getId().equals(root.getId()));
        
        data.put("files", getChildDocumentsData(folderId, true, false));
        data.put("children", getChildDocumentsData(folderId, false, true));
        
        return data;
    }
    
    
    /**
     * Retrieves the children of a document and extracts its data.
     * @param parentId Identifier of the parent collection. Can be null to get children of root folder.
     * @param excludeFolders Folders will be excluded if true
     * @param excludeFiles Files will be excluded if true
     * @return The map of information
     */
    @Callable
    public List<Map<String, Object>> getChildDocumentsData(String parentId, boolean excludeFolders, boolean excludeFiles)
    {
        ResourceCollection document = _getRootIfNull(parentId);
        if (document == null)
        {
            throw new IllegalArgumentException("Unable to get child documents: parent folder not found");
        }
        return getChildDocumentsData(document, excludeFolders, excludeFiles);
    }
    
    /**
     * Retrieves the children of a document and extracts its data.
     * @param document the document
     * @param excludeFolders Folders will be excluded if true
     * @param excludeFiles Files will be excluded if true
     * @return The map of information
     */
    public List<Map<String, Object>> getChildDocumentsData(ResourceCollection document, boolean excludeFolders, boolean excludeFiles)
    {
        ResourceCollection parent = _getRootIfNull(document);
        if (parent == null)
        {
            throw new IllegalArgumentException("Unable to get child documents: parent folder not found");
        }
        
        return parent.getChildren().stream()
            .map(child -> _extractDocumentData(child, excludeFolders, excludeFiles))
            .filter(Objects::nonNull)
            .collect(Collectors.toList());
    }
    
    /**
     * Retrieves the set of standard data for a document (folder or resource)
     * @param id the document id or null to get root document
     * @param excludeFilesInFolderHierarchy Should child files be taken into account when extracting data of a folder
     * @return The map of data
     */
    // TODO To remove not used
    @Deprecated
    @Callable
    public Map<String, Object> getDocumentData(String id, boolean excludeFilesInFolderHierarchy)
    {
        AmetysObject document = id == null ? _getRootFromRequest() : _resolver.resolveById(id);
        if (document == null)
        {
            throw new IllegalArgumentException("No project found in request to get root document data");
        }
        
        Map<String, Object> data = _extractDocumentData(document, false, false);
        return data != null ? data : new HashMap<>();
    }
    
    /**
     * Retrieves the set of standard data for a list of documents
     * @param ids The list of document identifiers
     * @param excludeFilesInFolderHierarchy Should child files be taken into account when extracting data of a folder 
     * @return The map of data
     */
    // TODO To remove not used
    @Callable
    public List<Map<String, Object>> getDocumentsData(List<String> ids, boolean excludeFilesInFolderHierarchy)
    {
        return ids.stream()
            .map(id -> getDocumentData(id, excludeFilesInFolderHierarchy))
            .collect(Collectors.toList());
    }
    
    /**
     * Generates an uri to open a document through webdav
     * @param documentId The document identifier
     * @return The generated uri
     */
    @Callable
    public String generateWebdavUri(String documentId)
    {
        return ResolveURIComponent.resolve("webdav-project-resource", documentId, false, true);
    }
    
    private ResourceCollection _getRootIfNull(ResourceCollection document)
    {
        return document != null ? document : _getRootFromRequest();
    }
    
    private ResourceCollection _getRootIfNull(String documentId)
    {
        return StringUtils.isNotEmpty(documentId)
                ? (ResourceCollection) _resolver.resolveById(documentId)
                : _getRootFromRequest();
    }
    
    private ResourceCollection _getRootFromRequest()
    {
        Project project = _getProjectFromRequest();
        
        if (project != null)
        {
            return _getRootFromProject(project);
        }
        else
        {
            return null;
        }
    }
    
    private ResourceCollection _getRootFromObject(AmetysObject ametysObject)
    {
        Project project = _getProjectFomObject(ametysObject);
        if (project != null)
        {
            return _getRootFromProject(project);
        }
        return null;
    }
    
    private ResourceCollection _getRootFromProject(Project project)
    {
        if (project != null)
        {
            DocumentWorkspaceModule module = _moduleEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
            return module.getModuleRoot(project, false);
        }
        else
        {
            return null;
        }
    }
    
    private Project _getProjectFromRequest()
    {
        Request request = ContextHelper.getRequest(_context);
        
        String projectName = (String) request.getAttribute("projectName");
        if (projectName != null)
        {
            return _projectManager.getProject(projectName);
        }
        else
        {
            return null;
        }
    }
    
    private Project _getProjectFomObject(AmetysObject ametysObject)
    {
        AmetysObject parent = ametysObject;
        
        while (parent != null && !(parent instanceof Project))
        {
            parent = parent.getParent();
        }
        
        if (parent == null)
        {
            return null;
        }
        return (Project) parent;
    }
    
    private String _getCurrentLanguage()
    {
        Request request = ContextHelper.getRequest(_context);
        return (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
    }
    
    /**
     * Internal method to extract the valuable data of a document
     * @param document The document (file or folder)
     * @param excludeFolders Folders will be excluded if true
     * @param excludeFiles Files will be excluded if true
     * @return the valuable document data
     */
    protected Map<String, Object> _extractDocumentData(AmetysObject document, boolean excludeFolders, boolean excludeFiles)
    {
        if (!excludeFiles && document instanceof Resource)
        {
            Resource file = (Resource) document;
            if (_canView(file))
            {
                return _extractFileData(file);
            }
        }
        else if (!excludeFolders && document instanceof ResourceCollection)
        {
            ResourceCollection folder = (ResourceCollection) document;
            if (_canView(folder))
            {
                return _extractFolderData(folder);
            }
        }
        
        return null;
    }
    
    /**
     * Internal method to extract the valuable data of a folder
     * @param folder The folder
     * @return the valuable folder data
     */
    protected Map<String, Object> _extractFolderData(ResourceCollection folder)
    {
        Map<String, Object> data = new HashMap<>();
        
        data.put("id", folder.getId());
        data.put("name", folder.getName());
        data.put("path", _getFolderPath(folder));
        data.put("type", ResourceType.FOLDER.name().toLowerCase());
        data.put("description", StringUtils.defaultString(folder.getDescription()));
        
        AmetysObject parent = folder.getParent();
        if (parent != null && parent instanceof ResourceCollection)
        {
            data.put("location", ((ResourceCollection) parent).getName());
            data.put("parentId", parent.getId());
        }
        
        boolean hasChildren = _hasChildren(folder, true);
        if (hasChildren)
        {
            data.put("children", Collections.EMPTY_LIST);
        }
        
        data.put("modifiable", folder instanceof ModifiableAmetysObject);
        data.put("canCreateChild", folder instanceof ModifiableExplorerNode);
        data.put("rights", _extractFolderRightData(folder));

        data.put("notification", false); // TODO unread notification (not yet supported)
        
        return data;
    }
    
    private List<String> _getFolderPath(ResourceCollection folder)
    {
        List<String> paths = new ArrayList<>();
        
        ResourceCollection rootDocuments = _getRootFromObject(folder);
        
        if (!rootDocuments.equals(folder))
        {
            List<ExplorerNode> parents = new ArrayList<>();
            
            AmetysObject parent = folder.getParent();
            while (parent instanceof ExplorerNode && !parent.equals(rootDocuments))
            {
                parents.add((ExplorerNode) parent);
                parent = parent.getParent();
            }
            
            parents.add(rootDocuments);
            
            Collections.reverse(parents);
            
            parents.stream().forEach(p -> 
            {
                paths.add(p.getId());
            });
        }
        
        return paths;
    }
    
    /**
     * Internal method to detect if a document has child
     * @param folder The folder
     * @param ignoreFiles Should child files be taken into account to compute the 'hasChildDocuments' data.
     * @return the valuable folder data
     */
    protected boolean _hasChildren(ResourceCollection folder, boolean ignoreFiles)
    {
        try (AmetysObjectIterable<AmetysObject> children = folder.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof ResourceCollection && _canView((ResourceCollection) child)
                    || !ignoreFiles && child instanceof Resource && _canView((Resource) child))
                {
                    return true;
                }
            }
            
            return false;
        }
    }
    
    /**
     * Internal method to extract the data concerning the right of the current user for a folder
     * @param folder The folder
     * @return The map of right data. Keys are the rights id, and values indicates whether the current user has the right or not.
     */
    protected  Map<String, Object> _extractFolderRightData(ResourceCollection folder)
    {
        Map<String, Object> rightsData = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        // Add
        rightsData.put("add-file", _rightManager.hasRight(user, RIGHTS_RESOURCE_ADD, folder) == RightResult.RIGHT_ALLOW);
        rightsData.put("add-folder", _rightManager.hasRight(user, RIGHTS_COLLECTION_ADD, folder) == RightResult.RIGHT_ALLOW);
        rightsData.put("add-cmis-folder", _rightManager.hasRight(user, "Plugin_Explorer_CMIS_Add", folder) == RightResult.RIGHT_ALLOW);
        
        // Rename - Edit
        rightsData.put("edit", _rightManager.hasRight(user, RIGHTS_COLLECTION_EDIT, folder) == RightResult.RIGHT_ALLOW);
        
        // Delete
        rightsData.put("delete", _rightManager.hasRight(user, RIGHTS_COLLECTION_DELETE, folder) == RightResult.RIGHT_ALLOW);
        // FIXME Delete own?
        
        return rightsData;
    }
    
    /**
     * Add a file
     * @param part The uploaded part corresponding to the file
     * @param parentId Identifier of the parent collection
     * @param unarchive True if the file is an archive that should be unarchived (only available for ZIP file)
     * @param allowRename True if the file can be renamed if it already exists
     * @param allowUpdate True if the file can be updated if it already exists (and allowRename is false)
     * @return The result map with id, parentId and name keys
     */
    @Callable
    public Map<String, Object> addFile(Part part, String parentId, boolean unarchive, boolean allowRename, boolean allowUpdate)
    {
        ModifiableResourceCollection modifiableFolder = getModifiableResourceCollection(parentId);
        _addOrUpdateResourceHelper.checkAddResourceRight(modifiableFolder);
        
        ResourceOperationMode mode = getOperationMode(unarchive, allowRename, allowUpdate);
        
        ResourceOperationResult operationResult = _addOrUpdateResourceHelper.performResourceOperation(part, modifiableFolder, mode);
        
        // Handle result map
        return generateActionResult(modifiableFolder, operationResult);
    }
    /**
     * Add a file
     * @param inputStream The uploaded input stream
     * @param fileName desired file name
     * @param parentId Identifier of the parent collection
     * @param unarchive True if the file is an archive that should be unarchived (only available for ZIP file)
     * @param allowRename True if the file can be renamed if it already exists
     * @param allowUpdate True if the file can be updated if it already exists (and allowRename is false)
     * @return The result map with id, parentId and name keys
     */
    public Map<String, Object> addFile(InputStream inputStream, String fileName, String parentId, boolean unarchive, boolean allowRename, boolean allowUpdate)
    {
        ModifiableResourceCollection modifiableFolder = getModifiableResourceCollection(parentId);
        ResourceOperationMode mode = getOperationMode(unarchive, allowRename, allowUpdate);
        
        ResourceOperationResult operationResult = _addOrUpdateResourceHelper.performResourceOperation(inputStream, fileName, modifiableFolder, mode);
        
        // Handle result map
        return generateActionResult(modifiableFolder, operationResult);
    }
    
    /**
     * get a {@link ModifiableResourceCollection} for an ID, or the root folder;
     * @param ametysId id of the resource. Can be null to get root folder.
     * @return ModifiableResourceCollection
     * @throws IllegalClassException if id links to a node which is not a {@link ModifiableResourceCollection}
     */
    private ModifiableResourceCollection getModifiableResourceCollection(String ametysId)
    {
        ResourceCollection folder = _getRootIfNull(ametysId);
        
        if (folder == null)
        {
            throw new IllegalArgumentException("Root folder not found");
        }
        
        if (!(folder instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, folder.getClass());
        }
        
        return (ModifiableResourceCollection) folder;
    }
    /**
     * returns the {@link ResourceOperationMode} according to parameters
     * @param unarchive unarchive
     * @param allowRename allowRename
     * @param allowUpdate allowUpdate
     * @return ADD, ADD_UNZIP, ADD_RENAME, ADD_UPDATE
     */
    private ResourceOperationMode getOperationMode(boolean unarchive, boolean allowRename, boolean allowUpdate)
    {
        ResourceOperationMode mode = ResourceOperationMode.ADD;
        if (unarchive)
        {
            mode = ResourceOperationMode.ADD_UNZIP;
        }
        else if (allowRename)
        {
            mode = ResourceOperationMode.ADD_RENAME;
        }
        else if (allowUpdate)
        {
            mode = ResourceOperationMode.UPDATE;
        }
        return mode;
    }
    private Map<String, Object> generateActionResult(ResourceCollection folder, ResourceOperationResult operationResult)
    {
        Map<String, Object> result = new HashMap<>();
        
        if (operationResult.isSuccess())
        {
            List<Map<String, Object>> resourceData = operationResult.getResources()
                    .stream()
                    .filter(r -> r.getParent().equals(folder)) // limit to direct children
                    .map(this::_extractFileData)
                    .collect(Collectors.toList());
            
            result.put("resources", resourceData);
            result.put("unzip", operationResult.isUnzip());
        }
        else
        {
            result.put("message", operationResult.getErrorMessage());
        }
        
        return result;
    }
    
    /**
     * Internal method to extract the valuable data of a file
     * @param file The file
     * @return the valuable file data
     */
    protected  Map<String, Object> _extractFileData(Resource file)
    {
        Map<String, Object> data = new HashMap<>();
        
        data.put("id", file.getId());
        data.put("name", file.getName());
        data.put("path", _getFilePath(file));
        
        // Encode path without extension
        String resourcePath = file.getResourcePath();
        int i = resourcePath.lastIndexOf(".");
        resourcePath = i != -1 ? resourcePath.substring(0, i) : resourcePath; 
        // Encode twice
        String encodedPath = FilenameUtils.encodePath(resourcePath);
        data.put("encodedPath", URIUtils.encodeURI(encodedPath, Map.of()));
        
        data.put("type", ResourceType.FILE.name().toLowerCase());
        data.put("fileType", _workspaceHelper.getFileType(file).name().toLowerCase());
        data.put("fileExtension", StringUtils.substringAfterLast(file.getName(), "."));
        
        AmetysObject parent = file.getParent();
        if (parent != null && parent instanceof ResourceCollection)
        {
            data.put("location", ((ResourceCollection) parent).getName());
            data.put("parentId", parent.getId());
            data.put("parentPath", ((ResourceCollection) parent).getExplorerPath());
        }
        
        data.put("modifiable", file instanceof ModifiableResource);
        data.put("canCreateChild", file instanceof ModifiableExplorerNode);
        
        data.put("description", file.getDCDescription());
        data.put("tags", _tags2json(file));
        data.put("mimetype", file.getMimeType());
        data.put("length", String.valueOf(file.getLength()));
        
        boolean image = _workspaceHelper.isImage(file);
        if (image)
        {
            data.put("image", true);
        }
        
        data.put("hasOnlyOfficePreview", _onlyOfficeManager.canBePreviewed(file.getId()));
        
        UserIdentity creatorIdentity = file.getCreator();
        data.put("creator", _userHelper.user2json(creatorIdentity));
        data.put("creationDate", DateUtils.dateToString(file.getCreationDate()));
        
        UserIdentity contribIdentity = file.getLastContributor();
        data.put("author", _userHelper.user2json(contribIdentity));
        data.put("lastModified", DateUtils.dateToString(file.getLastModified()));
        
        data.put("rights", _extractFileRightData(file));
        
        data.putAll(_extractFileLockData(file));
        
        return data;
    }
    
    private List<Map<String, Object>> _tags2json(Resource file)
    {
        return ((TagAwareAmetysObject) file).getTags()
            .stream()
            .filter(tag -> _tagProviderExtensionPoint.hasTag(tag, Map.of()))
            .map(tag -> _tagProviderExtensionPoint.getTag(tag, Map.of()))
            .map(this::_tag2json)
            .collect(Collectors.toList());
    }
    
    private Map<String, Object> _tag2json(Tag tag)
    {
        Map<String, Object> tagMap = new HashMap<>();
        tagMap.put("text", tag.getTitle());
        tagMap.put("name", tag.getName());
        tagMap.put("color", null); // FIXME tag color is not supported yet
        
        return tagMap;
    }
    
    private List<String> _getFilePath(Resource file)
    {
        return _getFolderPath(file.getParent());
    }
    
    /**
     * Internal method to extract the data concerning the right of the current user for file
     * @param file The file
     * @return The map of right data. Keys are the rights id, and values indicates whether the current user has the right or not.
     */
    protected  Map<String, Object> _extractFileRightData(Resource file)
    {
        Map<String, Object> rightsData = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        ResourceCollection folder = file.getParent();
        
        // Rename - Edit
        rightsData.put("rename", _rightManager.hasRight(user, RIGHTS_RESOURCE_RENAME, folder) == RightResult.RIGHT_ALLOW);
        rightsData.put("edit", _rightManager.hasRight(user, RIGHTS_RESOURCE_EDIT_DC, folder) == RightResult.RIGHT_ALLOW);
        
        // Delete
        rightsData.put("delete", _rightManager.hasRight(user, RIGHTS_RESOURCE_DELETE, folder) == RightResult.RIGHT_ALLOW);
        
        // TODO Delete own - no ability to detect document creator currently
        // rightsData.put("delete-own", ...);
        
        // Unlock
        rightsData.put("unlock", _rightManager.hasRight(user, RIGHTS_RESOURCE_UNLOCK_ALL, folder) == RightResult.RIGHT_ALLOW);
       
        // Comments
        rightsData.put("comment", _rightManager.hasRight(user, RIGHTS_RESOURCE_COMMENT, folder) == RightResult.RIGHT_ALLOW);
        rightsData.put("moderate-comments", _rightManager.hasRight(user, RIGHTS_RESOURCE_MODERATE_COMMENT, folder) == RightResult.RIGHT_ALLOW);
        
        return rightsData;
    }
    
    /**
     * Internal method to extract the data relative to the lock state of a file
     * @param file The file
     * @return The image specific data
     */
    protected  Map<String, Object> _extractFileLockData(Resource file)
    {
        Map<String, Object> lockData = new HashMap<>();
        
        if (file instanceof LockableAmetysObject)
        {
            boolean isLocked = ((LockableAmetysObject) file).isLocked();
            lockData.put("locked", isLocked);
            
            if (isLocked)
            {
                UserIdentity lockOwner = ((LockableAmetysObject) file).getLockOwner();
                
                lockData.put("isLockOwner", lockOwner.equals(_currentUserProvider.getUser()));
                lockData.put("lockOwner", _userHelper.user2json(lockOwner));
            }
        }
        
        return lockData;
    }
    
    private List<String> _sanitizeFileTags(Collection<String> tags) throws AmetysRepositoryException
    {
        // Enforce lowercase and remove possible duplicate tags
        return Optional.ofNullable(tags).orElseGet(ArrayList::new).stream()
                .map(String::trim)
                .map(String::toLowerCase)
                .distinct()
                .collect(Collectors.toList());
    }
    
    @Override
    protected void _setComment(JCRPost comment, String content)
    {
        try
        {
            _htmlTransformer.transform(content, comment.getContent());
        }
        catch (IOException e)
        {
            throw new AmetysRepositoryException("Failed to transform comment into rich text", e);
        }
    }
    
    @Override
    protected String _getComment(JCRPost post) throws AmetysRepositoryException
    {
        Source contentSource = null;
        try
        {
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("source", post.getContent().getInputStream());
            contentSource = _sourceResolver.resolveURI("cocoon://_plugins/" + _pluginName + "/convert/html2html", null, parameters);
            return IOUtils.toString(contentSource.getInputStream(), "UTF-8");
        }
        catch (IOException e)
        {
            throw new AmetysRepositoryException("Failed to transform rich text into string", e);
        }
        finally
        {
            _sourceResolver.release(contentSource);
        }
    }
    
    @Override
    protected String _getCommentForEditing(JCRPost post) throws AmetysRepositoryException
    {
        try
        {
            StringBuilder sb = new StringBuilder();
            _htmlTransformer.transformForEditing(post.getContent(), sb);
            return sb.toString();
        }
        catch (IOException e)
        {
            throw new AmetysRepositoryException("Failed to transform rich text into string", e);
        }
    }
    
    /**
     * Indicates if the current user can view the folder
     * @param folder The folder to test
     * @return true if the folder can be viewed
     */
    protected boolean _canView(ResourceCollection folder)
    {
        return _rightManager.currentUserHasReadAccess(folder);
    }
    
    /**
     * Indicates if the current user can view the file
     * @param file The file to test
     * @return true if the file can be viewed 
     */
    protected boolean _canView(Resource file)
    {
        return _rightManager.currentUserHasReadAccess(file.getParent());
    }
    
    @Override
    @Callable
    public Map<String, String> getCMISProperties(String id)
    {
        // override to allow calls from workspaces
        return super.getCMISProperties(id);
    }
    
    @Override
    @Callable
    public Map<String, Object> addCMISCollection(String parentId, String originalName, String url, String login, String password, String repoId, String mountPoint, boolean renameIfExists)
    {
        String rootId = parentId == null ? _getRootFromRequest().getId() : parentId;
        if (rootId == null)
        {
            throw new IllegalArgumentException("Unable to add CMIS collection: parent folder not found.");
        }
        return super.addCMISCollection(rootId, originalName, url, login, password, repoId, mountPoint, renameIfExists);
    }
    
    /**
     * Edits a CMIS folder (see {@link CMISRootResourcesCollection})
     * 
     * @param id the id of CMIS folder
     * @param name The name of the CMIS folder
     * @param url The url of CMIS repository
     * @param login The user's login to access CMIS repository
     * @param password The user's password to access CMIS repository
     * @param repoId The id of CMIS repository
     * @param mountPoint The mount point to use for the repository
     * @return the result map with id of edited node
     * @throws RepositoryException If an error occurred
     */
    @Callable
    public Map<String, Object> editCMISCollection(String id, String name, String url, String login, String password, String repoId, String mountPoint) throws RepositoryException
    {
        List<String> errors = new LinkedList<>();
        try
        {
            renameObject(id, name);
        }
        catch (RepositoryException e)
        {
            getLogger().error("Repository exception during CMIS folder edition.", e);
            errors.add("repository");
        }
        
        // override to allow calls from workspaces
        Map<String, Object> result = super.editCMISCollection(id, url, login, password, repoId, mountPoint);
        if (errors.size() > 0)
        {
            result.put("errors", errors);
        }
        return result;
    }
    
    @Override
    @Callable
    public boolean isCMISCollection(String id)
    {
        // override to allow calls from workspaces
        return super.isCMISCollection(id);
    }
    
    /**
     * Count the total of documents in the project
     * @param project The project
     * @return The total of documents, or null if the module is not activated
     */
    public Long getDocumentsCount(Project project)
    {
        Function<Project, ResourceCollection> getModuleRoot = proj -> _moduleEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID).getModuleRoot(proj, false);
        return Optional.ofNullable(project)
                .map(getModuleRoot)
                .map(root -> _getChildDocumentsCount(root))
                .orElse(null);
    }

    private Long _getChildDocumentsCount(ResourceCollection collection)
    {
        return collection.getChildren().stream()
                // Count the number of documents : a document counts as 1, a folder count has the sum of its children. Anything else is ignored (0)
                .map(ao -> ao instanceof Resource ? 1L : ao instanceof ResourceCollection ? _getChildDocumentsCount((ResourceCollection) ao) : 0L)
                .reduce(0L, Long::sum);
    }
    
    /**
     * Create a file 
     * @param folderId the folder parent id
     * @param type the type of file (word, excel, powerpoint, ...)
     * @return the new file properties
     * @throws Exception if an error occurred
     */
    @Callable
    public Map<String, Object> createFile(String folderId, String type) throws Exception
    {
        OfficeType officeType = OfficeType.valueOf(type.toUpperCase());

        String lang = _getCurrentLanguage();
        
        OpcPackage opcPackage = null;
        String name = null;
        String title = null;
        switch (officeType)
        {
            case EXCEL:
                opcPackage = SpreadsheetMLPackage.createPackage();
                ((SpreadsheetMLPackage) opcPackage).createWorksheetPart(
                        new PartName("/xl/worksheets/sheet1.xml"), 
                        _i18nUtils.translate(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EXCEL_NEW_FILE_PART1_TITLE"), lang),
                        1);
                name = "newExcel.xlsx";
                title = _i18nUtils.translate(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_EXCEL_NEW_FILE_TITLE"), lang) + ".xlsx";
                break;
            case WORD:
                opcPackage = WordprocessingMLPackage.createPackage();
                name = "newWord.docx";
                title = _i18nUtils.translate(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_WORD_NEW_FILE_TITLE"), lang) + ".docx";
                break;
            case POWERPOINT:
                opcPackage = PresentationMLPackage.createPackage();
                name = "newPowerPoint.pptx";
                title = _i18nUtils.translate(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_FILE_MANAGER_ONLYOFFICE_POWERPOINT_NEW_FILE_TITLE"), lang) + ".pptx";
                break;
            default:
                throw new IllegalArgumentException("Can create new file with unknown type '" + type + "'");
        }
        
        File file = new File(AmetysHomeHelper.getAmetysHomeData(), name);
        opcPackage.save(file);
        try (InputStream is = new FileInputStream(file))
        {
            String uniqueTitle = _getUniqueTitle(folderId, title);
            Map<String, Object> response = addFile(is, uniqueTitle, folderId, false, false, false);
            
            FileUtils.forceDelete(file);
            
            return response;
        }
    }

    private String _getUniqueTitle(String folderId, String title) throws RepositoryException
    {
        JCRAmetysObject folder = _resolver.resolveById(folderId);
        
        String ext = StringUtils.substringAfterLast(title, ".");
        String name = StringUtils.substringBeforeLast(title, ".");
        String newTitle = title;
        int count = 1;
        Node node = folder.getNode();
        while (node.hasNode(newTitle))
        {
            newTitle = name + " (" + count + ")." + ext;
            count++;
        }
        
        return newTitle;
    }
    
    @Override
    protected Map<String, Object> _comment2json(JCRPost comment, boolean isEdition)
    {
        Map<String, Object> comment2json = super._comment2json(comment, isEdition);
        
        String lang = _getCurrentLanguage();
        String authorImgUrl = _workspaceHelper.getAvatar(comment.getAuthor(), lang, 30);
        @SuppressWarnings("unchecked")
        Map<String, Object> author = (Map<String, Object>) comment2json.get("author");
        
        UserIdentity authorIdentity = comment.getAuthor();
        User user = _userManager.getUser(authorIdentity);
        if (user != null)
        {
            author.put("name", author.get("fullname"));
            author.put("avatar", authorImgUrl);
            author.put("id", UserIdentity.userIdentityToString(authorIdentity));
            Content member = _projectMemberManager.getUserContent(lang, user);
            if (member != null)
            {
                if (member.hasValue("function"))
                {
                    author.put("function", member.getValue("function"));
                }
                if (member.hasValue("organisation-accronym"))
                {
                    author.put("organisationAcronym", member.getValue("organisation-accronym"));
                }
            }
        }
        
        comment2json.put("text", isEdition ? _getCommentForEditing(comment) : _getComment(comment));
        
        return comment2json;
    }
}
