/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum.json;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.RichText;
import org.ametys.cms.data.RichTextHelper;
import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.forum.Thread;
import org.ametys.plugins.workspaces.forum.WorkspaceThreadUserPreferencesDAO;
import org.ametys.plugins.workspaces.util.WorkspaceObjectJSONHelper;
import org.ametys.runtime.model.type.DataContext;
/**
 * Helper to convert forum's thread to JSON
 */
public class ThreadJSONHelper extends WorkspaceObjectJSONHelper implements Component
{
    /** The Avalon role */
    public static final String ROLE = ThreadJSONHelper.class.getName();

    /** The current user provider. */
    protected WorkspaceThreadUserPreferencesDAO _workspaceThreadUserPrefDAO;

    /** Rich text helper */
    protected RichTextHelper _richTextHelper;

    /** Workspace helper */
    protected WorkspacesHelper _workspaceHelper;

    /** Ametys resolver */
    protected AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workspaceThreadUserPrefDAO = (WorkspaceThreadUserPreferencesDAO) manager.lookup(WorkspaceThreadUserPreferencesDAO.ROLE);
        _richTextHelper = (RichTextHelper) manager.lookup(RichTextHelper.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }

    /**
     * Thread as JSON
     * @param thread the forum thread
     * @param lang the language
     * @param siteName the site name
     * @return the thread as JSON
     */
    public Map<String, Object> threadAsJSON(Thread thread, String lang, String siteName)
    {
        return threadAsJSON(thread, lang, siteName, true);
    }

    /**
     * Thread as JSON
     * @param thread the forum thread
     * @param lang the language
     * @param siteName the site name
     * @param parseCommentcontent True to parse comment content
     * @return the thread as JSON
     */
    public Map<String, Object> threadAsJSON(Thread thread, String lang, String siteName, boolean parseCommentcontent)
    {
        Map<String, Object> json = new HashMap<>();

        json.put(Thread.ATTRIBUTE_ID, thread.getId());
        json.put(Thread.ATTRIBUTE_TITLE, thread.getTitle());

        RichText richText = thread.getContent();

        try
        {
            StringBuilder result = new StringBuilder(2048);
            _richTextTransformer.transformForEditing(richText, DataContext.newInstance(), result);
            json.put(Thread.ATTRIBUTE_CONTENT_FOR_EDITING, result.toString());
            json.put(Thread.ATTRIBUTE_CONTENT_ABSTRACT, richTextToSimpleText(richText, 200));
            json.put(Thread.ATTRIBUTE_CONTENT_FOR_RENDERING, richTextToRendering(richText));
        }
        catch (AmetysRepositoryException | IOException e)
        {
            json.put(ATTRIBUTE_FOR_RICHTEXT_ERROR, true);
            getLogger().error("Unable to transform the rich text value into a string for abstract for thread " + thread.getId(), e);
        }

        json.put(Thread.ATTRIBUTE_CREATIONDATE, thread.getCreationDate());
        json.put(Thread.ATTRIBUTE_LASTMODIFIED, thread.getLastModified());
        json.put(Thread.ATTRIBUTE_AUTHOR, _userHelper.user2json(thread.getAuthor()));
        json.put(Thread.IS_AUTHOR, thread.getAuthor().equals(_currentUserProvider.getUser()));
        json.put(Thread.ATTRIBUTE_CATEGORY, thread.getCategory().name());
        json.put(Thread.ATTRIBUTE_TAGS, _getTags(thread, siteName));
        json.put(Thread.ATTRIBUTE_CLOSEAUTHOR, _userHelper.user2json(thread.getCloseAuthor()));
        json.put(Thread.ATTRIBUTE_CLOSEDATE, thread.getCloseDate());

        List<Map<String, Object>> attachments = thread.getAttachments()
                .stream()
                .map(this::_binaryToJson)
                .collect(Collectors.toList());
        json.put(Thread.ATTRIBUTE_ATTACHMENTS, attachments);
        json.put(Thread.ATTRIBUTE_LASTCONTRIBUTION, thread.getLastContribution());

        ZonedDateTime lastReadDate = _workspaceThreadUserPrefDAO.getLastReadDate(thread);

        List<RichTextComment> flattenComments = thread.getFlattenComments(true, true);
        json.put(Thread.ATTRIBUTE_COMMENTS_LENTGH, flattenComments.size());
        if (parseCommentcontent)
        {
            List<Map<String, Object>> commentsAsJSON = _commentsToJson(flattenComments, lang, siteName, lastReadDate, parseCommentcontent);
            json.put(Thread.ATTRIBUTE_COMMENTS, commentsAsJSON);
        }

        List<Map<String, Object>> uniqueAuthors = _getUniqueAuthors(flattenComments, lang);
        json.put(Thread.ATTRIBUTE_COMMENTS_AUTHORS, uniqueAuthors);

        json.put(Thread.HAS_UNOPENED_THREAD_NOTIFICATION, !_workspaceThreadUserPrefDAO.hasOpenedThread(thread));
        json.put(Thread.UNREAD_COMMENTS, _workspaceThreadUserPrefDAO.getUnreadCommentNumber(thread));

        RichTextComment acceptedComment = thread.getAcceptedAnswer();
        if (acceptedComment != null)
        {
            json.put(Thread.ACCEPTED_ANSWER_COMMENT_ID, _commentToJson(acceptedComment, lang, siteName, lastReadDate, true));
        }

        return json;
    }

    private List<Map<String, Object>> _getUniqueAuthors(List<RichTextComment> flattenComments, String lang)
    {
        HashSet<String> seen = new HashSet<>();
        List<RichTextComment> uniqueAuthorComments = new ArrayList<>(flattenComments);
        uniqueAuthorComments.removeIf(e -> !seen.add(e.getAuthorEmail()));
        List<Map<String, Object>> uniqueAuthors = uniqueAuthorComments.stream()
                .map(comment -> _authorToJSON(comment, _userManager.getUser(comment.getAuthor()), lang))
                .toList();
        return uniqueAuthors;
    }

}
