/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.accesscontroller;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.Profile;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Give the read right on modules if the user as any other profile on it.
 * This controller has no cache by itself on the underlying non cached ProfileAssignmentStorageExtensionPoint ; but as it works only on the read right, the top level cache in the  RightManager do the job
 */
public class ModuleAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    private static Pattern __MODULE_ROOT_PATH_PATTERN = Pattern.compile("^(/ametys:plugins/workspaces/projects/(?:[^/]+)/ametys-internal:resources/(?:[^/]+)).*$");
    
    /** The extension point for the profile assignment storages */
    protected ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageEP;
    /** the right helper */
    protected ProjectRightHelper _projectRightHelper;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The module extension point */
    protected WorkspaceModuleExtensionPoint _moduleEP;
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The profile DAO */
    protected RightProfilesDAO _profileDAO;
    /** The site manager */
    protected SiteManager _siteManager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _profileAssignmentStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
        _profileDAO = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
        _projectRightHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public boolean isSupported(Object object)
    {
        if (object instanceof AmetysObject)
        {
            String path = ((AmetysObject) object).getPath();
            return path.startsWith("/ametys:plugins/workspaces/projects/")
                    && path.contains("/ametys-internal:resources/");
        }
        else
        {
            return false;
        }
    }
    
    private boolean _isModuleActivated(ResourceCollection moduleRoot)
    {
        Project project = moduleRoot.getParent().getParent();
        
        WorkspaceModule module = _moduleEP.getModuleByName(moduleRoot.getName());
        if (module == null)
        {
            throw new IllegalStateException("Can not find module from the module root name '" + moduleRoot.getName() + "'");
        }
        
        return _projectManager.isModuleActivated(project, module.getId());
    }
    
    private ModifiableResourceCollection _getModuleRoot(Object object)
    {
        AmetysObject node = (AmetysObject) object;
        
        Matcher matcher = __MODULE_ROOT_PATH_PATTERN.matcher(node.getPath());
        if (matcher.matches())
        {
            String rootPath = matcher.group(1);
            if (node.getPath().equals(rootPath))
            {
                return (ModifiableResourceCollection) node;
            }
            else
            {
                return (ModifiableResourceCollection) _resolver.resolveByPath(rootPath);
            }
        }
        
        throw new IllegalArgumentException("Node " + node.getPath() + " is not a module path");
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        ModifiableResourceCollection root = _getModuleRoot(object);
        
        if (!_isModuleActivated(root))
        {
            return AccessResult.UNKNOWN;
        }
        else
        {
            return _profileAssignmentStorageEP.getPermissions(user, userGroups, _projectRightHelper.getProfilesIds(), root)
                    .values().stream()
                    .anyMatch(ar -> ar.toRightResult() == RightResult.RIGHT_ALLOW) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
    }
    
    @Override
    public AccessExplanation explainReadAccessPermission(UserIdentity user, Set<GroupIdentity> groups, Object object)
    {
        return _explain(_getModuleRoot(object), AccessResult.USER_ALLOWED, AccessResult.UNKNOWN,
            root -> {
                Set<String> grantingProfile = _profileAssignmentStorageEP.getPermissions(user, groups, _projectRightHelper.getProfilesIds(), root)
                        .entrySet().stream()
                        .filter(entry -> entry.getValue().toRightResult() == RightResult.RIGHT_ALLOW)
                        .map(Entry::getKey)
                        .collect(Collectors.toSet());
                return Pair.of(grantingProfile, Set.of());
            }
        );
    }

    private AccessExplanation _explain(ModifiableResourceCollection root, AccessResult grantedPermission, AccessResult deniedPermission, Function<ModifiableResourceCollection, Pair<Set<String>, Set<String>>> profileSupplier)
    {
        AccessResult permission;
        String i18nkey;
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        i18nParams.put("module", _moduleEP.getModuleByName(root.getName()).getModuleTitle());
        
        if (!_isModuleActivated(root))
        {
            permission = AccessResult.UNKNOWN;
            i18nkey = "PLUGINS_WORKSPACES_MODULE_ACCESS_CONTROLLER_DISABLED_MODULE_EXPLANATION";
        }
        else
        {
            Pair<Set<String>, Set<String>> assignedProfiles = profileSupplier.apply(root);
            if (!assignedProfiles.getRight().isEmpty())
            {
                permission = deniedPermission;
                i18nkey = "PLUGINS_WORKSPACES_MODULE_ACCESS_CONTROLLER_" + deniedPermission.name() + "_EXPLANATION";
                Set<Profile> denyingProfiles = assignedProfiles.getRight().stream().map(_profileDAO::getProfile).collect(Collectors.toSet());
                i18nParams.put("profiles", AccessExplanation.profilesToI18nizableText(denyingProfiles));
            }
            else if (!assignedProfiles.getLeft().isEmpty())
            {
                permission = grantedPermission;
                i18nkey = "PLUGINS_WORKSPACES_MODULE_ACCESS_CONTROLLER_" + grantedPermission.name() + "_EXPLANATION";
                Set<Profile> grantingProfiles = assignedProfiles.getLeft().stream().map(_profileDAO::getProfile).collect(Collectors.toSet());
                i18nParams.put("profiles", AccessExplanation.profilesToI18nizableText(grantingProfiles));
            }
            else
            {
                permission = AccessResult.UNKNOWN;
                i18nkey = "PLUGINS_WORKSPACES_MODULE_ACCESS_CONTROLLER_UNKNOWN_EXPLANATION";
            }
        }
        return new AccessExplanation(
                getId(),
                permission,
                new I18nizableText("plugin.workspaces", i18nkey, i18nParams)
                );
    }
    
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return Map.of();
    }
    
    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        ModifiableResourceCollection root = _getModuleRoot(object);
        
        if (!_isModuleActivated(root))
        {
            return AccessResult.UNKNOWN;
        }
        else
        {
            return _profileAssignmentStorageEP.getPermissionForAnonymous(_projectRightHelper.getProfilesIds(), root);
        }
    }
    
    @Override
    public AccessExplanation explainReadAccessPermissionForAnonymous(Object object)
    {
        return _explain(_getModuleRoot(object), AccessResult.ANONYMOUS_ALLOWED, AccessResult.ANONYMOUS_DENIED,
            root -> {
                Map<AnonymousOrAnyConnectedKeys, Set<String>> assignments = _profileAssignmentStorageEP.getProfilesForAnonymousAndAnyConnectedUser(root);
                
                Set<String> workspaceProfiles = _projectRightHelper.getProfilesIds();

                Set<String> deniedProfiles = assignments.get(AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED) .stream()
                        .filter(p -> workspaceProfiles.contains(p))
                        .collect(Collectors.toSet());
                
                
                Set<String> allowedProfiles = assignments.get(AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED).stream()
                        .filter(p -> workspaceProfiles.contains(p))
                        .collect(Collectors.toSet());
                
                return Pair.of(allowedProfiles, deniedProfiles);
            }
        );
    }
    
    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        ModifiableResourceCollection root = _getModuleRoot(object);
        
        if (!_isModuleActivated(root))
        {
            return AccessResult.UNKNOWN;
        }
        else
        {
            return _profileAssignmentStorageEP.getPermissionForAnyConnectedUser(_projectRightHelper.getProfilesIds(), root);
        }
    }
    
    @Override
    public AccessExplanation explainReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return _explain(_getModuleRoot(object), AccessResult.ANY_CONNECTED_ALLOWED, AccessResult.ANY_CONNECTED_DENIED,
            root -> {
                Map<AnonymousOrAnyConnectedKeys, Set<String>> assignments = _profileAssignmentStorageEP.getProfilesForAnonymousAndAnyConnectedUser(root);
                
                Set<String> workspaceProfiles = _projectRightHelper.getProfilesIds();
                Set<String> deniedProfiles = assignments.get(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED).stream()
                        .filter(p -> workspaceProfiles.contains(p))
                        .collect(Collectors.toSet());
                
                Set<String> allowedProfiles = assignments.get(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED).stream()
                        .filter(p -> workspaceProfiles.contains(p))
                        .collect(Collectors.toSet());
                
                return Pair.of(allowedProfiles, deniedProfiles);
            }
        );
    }
    
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return Map.of();
    }
    
    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        ModifiableResourceCollection root = _getModuleRoot(object);
        
        if (!_isModuleActivated(root))
        {
            return Map.of();
        }
        else
        {
            return _profileAssignmentStorageEP.getPermissionsByUser(_projectRightHelper.getProfilesIds(), root);
        }
    }
    
    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }
    
    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        ModifiableResourceCollection root = _getModuleRoot(object);
        
        if (!_isModuleActivated(root))
        {
            return Map.of();
        }
        else
        {
            return _profileAssignmentStorageEP.getPermissionsByGroup(_projectRightHelper.getProfilesIds(), root);
        }
    }
    
    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }
    
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false; // No need to forward to workspace
    }
    
    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false; // No need to forward to workspace
    }
    
    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false; // No need to forward to workspace
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult permission, Object object)
    {
        switch (permission)
        {
            case UNKNOWN:
                return new AccessExplanation(getId(), permission, new I18nizableText(
                        "plugin.workspaces",
                        "PLUGINS_WORKSPACES_MODULE_ACCESS_CONTROLLER_UNKNOWN_EXPLANATION",
                        Map.of("module", getObjectLabel(object))
                    ));
                
            default:
                return super.getStandardAccessExplanation(permission, object);
        }
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        return _moduleEP.getModuleByName(_getModuleRoot(object).getName()).getModuleTitle();
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return ProjectAccessController.WORKSPACE_CONTEXT_CATEGORY;
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        Site site = _siteManager.getSite(siteName);
        
        if (site != null)
        {
            List<Project> projects = _projectManager.getProjectsForSite(site);
            if (!projects.isEmpty())
            {
                try
                {
                    String query = "/jcr:root" + projects.get(0).getNode().getPath() + "/ametys-internal:resources/*";
                    return _resolver.query(query);
                }
                catch (RepositoryException e)
                {
                    getLogger().warn("Failed to list project modules", e);
                }
            }
        }
        return List.of();
    }
}
