/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.io.IOException;
import java.util.List;
import java.util.Locale;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.FullTextExpression;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.forum.ForumWorkspaceModule;
import org.ametys.plugins.workspaces.forum.jcr.JCRThread;
import org.ametys.plugins.workspaces.forum.json.ThreadJSONHelper;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Generator for threads search module
 */
public class ThreadSearchModuleGenerator extends AbstractXpathSearchModuleGenerator
{
    /** Thread Module */
    protected ForumWorkspaceModule _forumModule;
    /** The user directory */
    protected UserDirectoryHelper _userDirectoryHelper;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    /** The thread JSON helper */
    protected ThreadJSONHelper _threadJSONHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);

        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _forumModule = moduleManagerEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _threadJSONHelper = (ThreadJSONHelper) smanager.lookup(ThreadJSONHelper.ROLE);
    }

    @Override
    protected String getXPathQuery(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception
    {
        List<Project> projects = getProjects(request, true);

        List<Project> filteredProjects = filterProjectsForModule(projects, ForumWorkspaceModule.FORUM_MODULE_ID);
        if (filteredProjects.isEmpty())
        {
            // No project available for this module
            return null;
        }

        String projectXPathQuery = getProjectXPathQuery(filteredProjects);

        String searchQuery = "";
        if (StringUtils.isNotBlank(textfield))
        {
            Expression labelEXpr = new FullTextExpression("title", textfield);
            searchQuery = "[" + labelEXpr.build() + "]";
        }

        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion("creationDate", false, false);
        String sortQuery = sortCriteria.build();

        String jcrQuery = projectXPathQuery + "//forums/element(*, ametys:forum-thread)" + searchQuery  + " " + sortQuery;

        return jcrQuery;
    }

    @Override
    protected void saxHit(AmetysObject object, String lang) throws Exception
    {
        if (object instanceof JCRThread)
        {
            JCRThread thread = (JCRThread) object;
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", thread.getId());
            XMLUtils.startElement(contentHandler, "hit", attrs);

            XMLUtils.createElement(contentHandler, "title", thread.getTitle());
            if (thread.getContent() != null)
            {
                String simpleText = _threadJSONHelper.richTextToSimpleText(thread.getContent(), 150);
                XMLUtils.createElement(contentHandler, "description", simpleText);
            }

            Project project = getProject(object);
            String threadUri = _forumModule.getThreadUri(project, thread.getId());
            XMLUtils.createElement(contentHandler, "uri", threadUri);

            XMLUtils.createElement(contentHandler, "creationDate", DateUtils.zonedDateTimeToString(thread.getCreationDate()));

            XMLUtils.createElement(contentHandler, "category", thread.getCategory().toString());

            UserIdentity author = thread.getAuthor();
            saxUser(author, "author");

            saxAuthorInfos(author, lang);

            saxProject(project);

            XMLUtils.endElement(contentHandler, "hit");
        }
    }

    /**
     * SAX the author information
     * @param author the author
     * @param lang the current language
     * @throws SAXException if an error occurred
     * @throws IOException if an error occurred
     * @throws BadItemTypeException if an error occurred
     */
    protected void saxAuthorInfos(UserIdentity author, String lang) throws SAXException, BadItemTypeException, IOException
    {
        Content userContent = _userDirectoryHelper.getUserContent(author, lang);
        if (userContent != null)
        {
            AttributesImpl contentAttrs = new AttributesImpl();
            contentAttrs.addCDATAAttribute("id", userContent.getId());
            XMLUtils.startElement(contentHandler, "author-infos", contentAttrs);
            ContentType contentType = _cTypeEP.getExtension(WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
            View view = contentType.getView("abstract");
            userContent.dataToSAX(contentHandler, view, DataContext.newInstance().withLocale(new Locale(lang)).withEmptyValues(false));
            XMLUtils.endElement(contentHandler, "author-infos");
        }
    }

}
