/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.wall.observers;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.wall.WallContentModule;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * {@link Observer} to invalidate zone item cache of wall content service when a wall content is commented
 *
 */
public abstract class AbstractInvalidateZoneItemCacheOnWallObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The content type helper. */
    protected ContentTypesHelper _contentTypeHelper;

    /** The page element cache */
    protected PageElementCache _zoneItemCache;
    
    /** Workspaces project manager */
    protected ProjectManager _projectManager;
    
    /** Workspaces module manager */
    protected WorkspaceModuleExtensionPoint _moduleManagerEP;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _zoneItemCache = (PageElementCache) manager.lookup(PageElementCache.ROLE + "/zoneItem");
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
    }
    
    public int getPriority(Event event)
    {
        // processed just before front-office cache invalidation
        return MAX_PRIORITY + 3500;
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        if (content instanceof WebContent)
        {
            String siteName = ((WebContent) content).getSiteName();
            
            Optional<Project> optProject = _projectManager.getProjectsForSite(siteName)
                .stream()
                .findFirst()
                .map(p -> _projectManager.getProject(p));
            
            if (optProject.isPresent())
            {
                Project project = optProject.get();
                
                WorkspaceModule module = _moduleManagerEP.getModule(WallContentModule.WALLCONTENT_MODULE_ID);
                if (module != null && _projectManager.isModuleActivated(project, WallContentModule.WALLCONTENT_MODULE_ID))
                {
                    Optional<Page> optPage = _projectManager.getModulePages(project, module)
                        .stream()
                        .filter(p -> p.getSitemapName().equals(content.getLanguage()))
                        .findFirst();
                    
                    if (optPage.isPresent())
                    {
                        Page page = optPage.get();
                        Optional< ? extends ZoneItem> optZoneItem = page.getZone("default").getZoneItems()
                                .stream()
                                .filter(z -> z.getType() == ZoneType.SERVICE && z.getServiceId().equals(WallContentModule.WALLCONTENT_SERVICE_ID))
                                .findFirst();
                        
                        if (optZoneItem.isPresent())
                        {
                            ZoneItem zoneItem = optZoneItem.get();
                            _zoneItemCache.removeItem(null, siteName, "SERVICE:" + zoneItem.getServiceId(), zoneItem.getId());
                        }
                    }
                }
                
            }
        }
        
    }

}
