/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 import { callMethod } from '../../../helper/ServerCommHelper';
 import i18n from 'i18n';
 import AmetysFront from 'AmetysFront';

 export const actions = {

    /**
     * Add comment to a thread
     * @param   {String}  ametysObjectId  the id of the thread
     * @param   {String}  text  the text of the comment
     * @return  {Object} should return the comments
     */
    async addComment( {commit}, {ametysObjectId, text}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_ADD_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'createComment',
          parameters: [ametysObjectId, text]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_ADDED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_ADD_ERROR_MSG,
              text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

    /**
     * Edit comment of a thread
     * @param   {String}  ametysObjectId  the id of the thread
     * @param   {String}  commentId  the id of the comment
     * @param   {String}  text  the text of the comment
     * @return  {Object} should return the comments
     */
    async editComment({ commit }, {ametysObjectId, commentId, text}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_EDIT_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'editComment',
          parameters: [ametysObjectId, commentId, text]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_EDITED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_EDIT_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

    /**
     * Delete comment of a thread
     * @param   {String}  ametysObjectId  the id of the thread
     * @param   {String}  commentId  the id of the comment
     * @param   {Boolean} accepted  filter by whether threads have accepted answer
     * @return  {Object} should return the comments
     */
    async deleteComment({ commit }, {ametysObjectId, commentId, accepted}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_DELETE_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'deleteComment',
          parameters: [ametysObjectId, commentId, accepted]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_DELETED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_DELETE_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

     /**
      * Answer to a comment of a thread
      * @param   {String}  ametysObjectId  the id of the thread
      * @param   {String}  commentId  the id of the parent comment
      * @param   {String}  text  the text of the comment
      * @return  {Object} should return the comments
      */
    async addSubComment({ commit }, {ametysObjectId, commentId, text}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_ADD_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'answerComment',
          parameters: [ametysObjectId, commentId, text]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_ADDED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_ADD_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },


   /**
    * Like or unlike a thread comment
    * @param   {String}  ametysObjectId  the id of the thread
    * @param   {String}  commentId  the id of the parent comment
    * @param   {Boolean}  liked true if the comment is liked, otherwise the comment is unliked
    * @return  {Object} should return the comments
    */
    async likeOrUnlikeComment({ commit }, {ametysObjectId, commentId, liked}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_LIKE_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'likeOrUnlikeComment',
          parameters: [ametysObjectId, commentId, liked]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_LIKED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_LIKE_COMMENT_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

   /**
    * Accept a thread comment
    * @param   {String}  ametysObjectId  the id of the thread
    * @param   {String}  commentId  the id of the comment to accept
    * @param   {String}  oldCommentId old accepted comment id, if present. Used to unaccept the old accepted answer
    * @param   {Boolean} accepted  filter by whether threads have accepted answer
    * @return  {Object} should return the comments
    */
    async acceptComment({ commit }, {ametysObjectId, commentId, oldCommentId, accepted}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_ACCEPT_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'acceptComment',
          parameters: [ametysObjectId, commentId, oldCommentId, accepted]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_ACCEPTED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_ACCEPT_COMMENT_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },
   /**
    * Cancel a comment set as accepted answer
    * @param   {String}  ametysObjectId  the id of the thread
    * @param   {String}  commentId  the id of the parent comment
    * @param   {Boolean} accepted  filter by whether threads have accepted answer
    * @return  {Object} should return the comments
    */
    async cancelAcceptComment({ commit }, {ametysObjectId, commentId, accepted}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_CANCEL_ACCEPT_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'cancelAcceptComment',
          parameters: [ametysObjectId, commentId, accepted]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_CANCEL_ACCEPTD
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_CANCEL_ACCEPT_COMMENT_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

   /**
    * Report a comment
    * @param   {String}  ametysObjectId  the id of the thread
    * @param   {String}  commentId  the id of the comment
    * @return  {Object} should return the comments
    */
    async reportComment({ commit }, {ametysObjectId, commentId}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_REPORT_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'reportComment',
          parameters: [ametysObjectId, commentId]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_REPORT_ACCEPTED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_REPORT_COMMENT_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

   /**
    * Clear reports of a comment
    * @param   {String}  ametysObjectId  the id of the thread
    * @param   {String}  commentId  the id of the comment
    * @return  {Object} should return the comments
    */
    async clearReports({ commit }, {ametysObjectId, commentId}) {
      return new Promise(function(resolve, reject)
      {
        AmetysFront.Event.fire('loaderStart', {
          text: i18n.PLUGINS_WORKSPACES_FORUM_CLEAR_REPORT_COMMENT
        });

        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadCommentDAO',
          methodName: 'clearReports',
          parameters: [ametysObjectId, commentId]
        })
        .then(thread => {
          commit('thread/UPDATE_THREAD', thread, { root: true });
          resolve({comments: thread.comments});
          AmetysFront.Event.fire('loaderEnd', {
            text: i18n.PLUGINS_WORKSPACES_FORUM_COMMENT_CLEAR_REPORT_ACCEPTED
          });
        })
        .catch(error => {
          AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_CLEAR_REPORT_COMMENT_ERROR_MSG,
              text : i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
              details: error
          });
          reject();
        })
      });
    },

};
