/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import i18n from 'i18n';
import AmetysFront from 'AmetysFront';
import { callMethod } from '../../helper/ServerCommHelper';

export const namespaced = true;

export const state = {
  tasksLists: [],
  tasksListColors: [],
  tasksListIcons: []
};

export const mutations = {
  SET_TASKSLISTS(state, tasksLists) {
    state.tasksLists = tasksLists;
  },
  ADD_TASKLIST(state, tasksList) {
    state.tasksLists.push(tasksList);
  },
  DELETE_TASKLIST(state, index) {
    state.tasksLists.splice(index, 1);
  },
  UPDATE_TASKLIST(state, editedTasksList) {
    const index = state.tasksLists.findIndex(tasksList => tasksList.id === editedTasksList.id)

    if (index !== -1) {
      const taskList = state.tasksLists[index];
      for (const key in editedTasksList) {
        taskList[key] = editedTasksList[key];
      }
      state.tasksLists.splice(index, 1, taskList)
    }
  },
  SET_TASKSLIST_COLORS(state, tasksListColors) {
    state.tasksListColors = tasksListColors;
  },
  SET_TASKSLIST_ICONS(state, tasksListIcons) {
    state.tasksListIcons = tasksListIcons;
  } 
};

export const getters = {
  getTasksLists: (state) => {
    const colors = state.tasksListColors || [];
    const icons = state.tasksListIcons || [];
    const data = state.tasksLists.map((t) => {
      t.color = colors.find(c=> c.id == t.colorId) || null;
      t.icon = icons.find(c=> c.id == t.iconId) || null;
      return t; 
    }); 
    return data;
  },
  getTasksListColors: (state) => {
    return state.tasksListColors;
  },
  getTasksListIcons: (state) => {
    return state.tasksListIcons;
  },
}

export const actions = {
  async loadTasksLists({commit}) {
    AmetysFront.Event.fire('loader', true)

    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'getTasksLists',
      parameters: []
    })
    .then(data => {
      commit('SET_TASKSLISTS', data);
      AmetysFront.Event.fire('loader', false)
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASKS_LIST_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },

  async addTasksList({commit}, {label}) {
    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_ADD
    });
    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'addTasksList',
      parameters: [
        label
      ]
    })
    .then(data => {
      commit('ADD_TASKLIST', data);
      
      AmetysFront.Event.fire('loaderEnd', {
        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_ADD
      });
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },

  async deleteTasksList({commit, state}, {id}) {
    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_DELETE
    });
  
    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'deleteTasksList',
      parameters: [
        id
      ]
    })
    .then(() => {
      const index = state.tasksLists.findIndex(tasksList => tasksList.id == id)
      commit('DELETE_TASKLIST', index);
      
      AmetysFront.Event.fire('loaderEnd', {
        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_DELETE
      });
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_DELETE_TASKS_LIST_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },

  async editTasksList({commit}, {id, newLabel}) {
    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_EDIT
    });
  
    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'editTasksList',
      parameters: [
        id,
        newLabel
      ]
    })
    .then(data => {
      commit('UPDATE_TASKLIST', {id: data.id, label: newLabel});
  
      AmetysFront.Event.fire('loaderEnd', {
        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_EDIT
      });
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_EDIT_TASKS_LIST_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },
  
  async moveTasksList({commit}, {id, newPosition}) {
    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_MOVE
    });
  
    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'moveTasksList',
      parameters: [
        id,
        newPosition
      ]
    })
    .then(data => {
      commit('UPDATE_TASKLIST', {id: data.id, position: newPosition});
      AmetysFront.Event.fire('loaderEnd', {
        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_MOVE
      });
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_MOVE_TASKS_LIST_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },

  async loadTasksListColors({commit}) {
    AmetysFront.Event.fire('loader', true)

    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'getTasksListColors',
      parameters: []
    }) 
    .then(data => {
      commit('SET_TASKSLIST_COLORS', data);
      AmetysFront.Event.fire('loader', false)
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASKS_LIST_COLORS_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      }); 
    })
  },

  async updateTasksListColor({commit}, {id, colorId}) {
    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_COLOR_EDIT
    }); 
    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'updateTasksListColor',
      parameters: [
        id,
        colorId
      ]
    })
    .then(() => {
      commit('UPDATE_TASKLIST', { id, colorId });
      AmetysFront.Event.fire('loaderEnd', {
        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_COLOR_EDIT
      });
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_TASKS_LIST_COLOR_EDIT_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },

  async loadTasksListIcons({commit}) {
    AmetysFront.Event.fire('loader', true)

    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'getTasksListIcons',
      parameters: []
    }) 
    .then(data => {
      commit('SET_TASKSLIST_ICONS', data);
      AmetysFront.Event.fire('loader', false)
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASKS_LIST_ICONS_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      }); 
    })
  },

  async updateTasksListIcon({commit}, {id, iconId}) {
    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_ICON_EDIT
    });
    await callMethod({
      role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
      methodName: 'updateTasksListIcon',
      parameters: [
        id,
        iconId
      ]
    })
    .then(() => {
      commit('UPDATE_TASKLIST', { id, iconId });
      AmetysFront.Event.fire('loaderEnd', {
        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_ICON_EDIT
      });
    })
    .catch(error => {
      AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_TASKS_LIST_ICON_EDIT_ERROR_MSG,
          text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
          details: error
      });
    })
  },
}
