/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.activitystream;

import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.WebHelper;

/**
 * Get the activities for activity stream
 *
 */
public class ActivityStreamServiceAction extends ServiceableAction
{
    private ActivityStreamClientInteraction _activityStream;
    private ProjectManager _projectManager;
    private UserPreferencesManager _userPrefManager;
    private CurrentUserProvider _currentUserProvider;
    private WorkspaceModuleExtensionPoint _moduleEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _activityStream = (ActivityStreamClientInteraction) smanager.lookup(ActivityStreamClientInteraction.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _userPrefManager = (UserPreferencesManager) smanager.lookup(UserPreferencesManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) smanager.lookup(WorkspaceModuleExtensionPoint.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        int limit = parameters.getParameterAsInteger("max-results", Integer.MAX_VALUE);
        
        boolean allActivities = parameters.getParameterAsBoolean("all", false);
        boolean markAsRead = parameters.getParameterAsBoolean("mark-as-read", false);
        
        // The filtered modules
        Set<String> activityTypes = _getFilteredActivityTypes(request);
        
        // The filtered projects' categories
        Set<String> categories = _getFilteredCategories(request);
        
        // Fultext search
        String pattern = request.getParameter("pattern");
        
        // Dante range
        Date fromDate = null;
        String fromDateAsStr = request.getParameter("fromDate");
        if (StringUtils.isNotEmpty(fromDateAsStr))
        {
            fromDate = DateUtils.asDate(DateUtils.parseZonedDateTime(fromDateAsStr));
        }
        Date untilDate = null;
        String untilDateAsStr = request.getParameter("untilDate");
        if (StringUtils.isNotEmpty(untilDateAsStr))
        {
            untilDate = DateUtils.asDate(DateUtils.parseZonedDateTime(untilDateAsStr));
        }
        
        String siteName = WebHelper.getSiteName(request);
        
        List<String> projects = _projectManager.getProjectsForSite(siteName);
        
        List<Map<String, Object>> activities;
        if (projects.isEmpty() || allActivities)
        {
            // Get activity stream of all current user's projects
            activities = _activityStream.getActivitiesForCurrentUser(pattern, categories, activityTypes, fromDate, untilDate, limit);
        }
        else
        {
            // Get activity stream  of current project only
            Project project = _projectManager.getProject(projects.get(0));
            activities = _activityStream.getActivitiesForCurrentUser(Collections.singleton(project), activityTypes, fromDate, untilDate, pattern, limit);
        }
        
        UserIdentity user = _currentUserProvider.getUser();
        if (markAsRead && user != null)
        {
            // Mark as read in user prefs
            _userPrefManager.setUserPreferences(user, ActivityStreamClientInteraction.ACTIVITY_STREAM_USER_PREF_CONTEXT, Map.of(), Map.of(ActivityStreamClientInteraction.ACTIVITY_STREAM_USER_PREF_LAST_UPDATE, DateUtils.dateToString(new Date())));
        }
        
        Map<String, Object> result = new HashMap<>();
        // FIXME this should be rename activities
        result.put("events", activities);
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        
        return EMPTY_MAP;
    }
    
    private Set<String> _getFilteredActivityTypes(Request request)
    {
        String[] moduleIds = request.getParameterValues("modules");
        if (moduleIds != null)
        {
            return Arrays.stream(moduleIds)
                    .filter(StringUtils::isNotEmpty)
                    .map(id -> _moduleEP.<WorkspaceModule>getModule(id))
                    .map(WorkspaceModule::getAllowedEventTypes)
                    .flatMap(Set::stream)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }
    
    private Set<String> _getFilteredCategories(Request request)
    {
        String[] categoryIds = request.getParameterValues("categories");
        if (categoryIds != null)
        {
            return Arrays.stream(categoryIds)
                    .filter(StringUtils::isNotEmpty)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }

}
