/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.calendars;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.calendars.ObservationConstants;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * {@link ActivityType} implementation for the creation of a calendar event
 */
public abstract class AbstractCalendarEventActivityType extends AbstractWorkspacesActivityType
{
    /** Constant for event's category */
    public static final String ACTIVITY_CATEGORY_CALENDARS = "calendars";
    
    /** data name for the event title */
    public static final String CALENDAR_EVENT_TITLE = "eventTitle";
    /** data name for the event id */
    public static final String CALENDAR_EVENT_ID = "eventId";
    /** data name for the calendar title */
    public static final String CALENDAR_TITLE = "calendarTitle";
    /** data name for the calendar id */
    public static final String CALENDAR_ID = "calendarId";
    /** data name for the event start date */
    public static final String CALENDAR_EVENT_START_DATE = "eventStartDate";
    /** data name for the event end date */
    public static final String CALENDAR_EVENT_END_DATE = "eventEndDate";
    /** data name for the event full day */
    public static final String CALENDAR_EVENT_FULL_DATE = "eventFullDay";
    /** data name for the event recurrence */
    public static final String CALENDAR_EVENT_RECURRENCE = "eventRecurrence";
    /** data name for the event recurrence end date */
    public static final String CALENDAR_EVENT_REPEATER_UNTIL = "eventRepeatUntil";

    private CalendarWorkspaceModule _calendarModule;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _calendarModule = moduleManagerEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
    }
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        activity.setValue(PROJECT_CATEGORY, ACTIVITY_CATEGORY_CALENDARS);
        
        Calendar calendar = (Calendar) parameters.get(ObservationConstants.ARGS_CALENDAR);
        CalendarEvent calendarEvent = (CalendarEvent) parameters.get(ObservationConstants.ARGS_CALENDAR_EVENT);
        
        // Calendar
        activity.setValue(CALENDAR_TITLE, calendar.getName());
        activity.setValue(CALENDAR_ID, calendar.getId());
        
        // Event
        activity.setValue(CALENDAR_EVENT_TITLE, calendarEvent.getTitle());
        activity.setValue(CALENDAR_EVENT_ID, calendarEvent.getId());
        activity.setValue(CALENDAR_EVENT_START_DATE, calendarEvent.getStartDate());
        activity.setValue(CALENDAR_EVENT_END_DATE, calendarEvent.getEndDate());
        activity.setValue(CALENDAR_EVENT_FULL_DATE, calendarEvent.getFullDay());
        activity.setValue(CALENDAR_EVENT_RECURRENCE, calendarEvent.getRecurrenceType().name().toUpperCase());
        if (!calendarEvent.getRecurrenceType().equals(EventRecurrenceTypeEnum.NEVER) && calendarEvent.getRepeatUntil() != null)
        {
            activity.setValue(CALENDAR_EVENT_REPEATER_UNTIL, DateUtils.zonedDateTimeToString(calendarEvent.getRepeatUntil()));
        }
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        
        String calendarId = activity.getValue(CALENDAR_ID);
        String calendarEventId = activity.getValue(CALENDAR_EVENT_ID);
        
        String projectName = activity.getValue(PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            if (!ObservationConstants.EVENT_CALENDAR_EVENT_DELETING.equals(activity.getEventType()))
            {
                json.put("eventUrl", _calendarModule.getEventUri(project, calendarId, calendarEventId));
            }
            else
            {
                json.put("eventUrl", _calendarModule.getModuleUrl(project));
            }
        }

        return json;
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        Expression eventExpr = new StringExpression(CALENDAR_EVENT_TITLE, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
        Expression calExpr = new StringExpression(CALENDAR_TITLE, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
        return new OrExpression(eventExpr, calExpr);
    }
    
    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        if (!super.isMergeable(activity1, activity2))
        {
            return false;
        }
        String calendarId1 = activity1.getValue(CALENDAR_ID);
        String calendarId2 = activity2.getValue(CALENDAR_ID);
        
        // FIXME GG pourquoi on ne prends pas l'event en compte ?
        
        return calendarId1 != null && calendarId2 != null && calendarId1.equals(calendarId2);
    }
    
    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivities = super.mergeActivities(activities);
        
        List<Map<String, Object>> mergedEvents = new ArrayList<>();
        
        List<String> knownEvents = new ArrayList<>();
        
        for (Activity activity : activities)
        {
            Map<String, Object> eventInfo = new HashMap<>();
            
            String eventId = activity.getValue(CALENDAR_EVENT_ID);
            if (!knownEvents.contains(eventId))
            {
                knownEvents.add(eventId);
                Map<String, Object> jsonActivity = activity.toJSONForClient();
                
                eventInfo.put(CALENDAR_EVENT_ID, eventId);
                eventInfo.put(CALENDAR_EVENT_TITLE, jsonActivity.get(CALENDAR_EVENT_TITLE));
                eventInfo.put(CALENDAR_EVENT_START_DATE, jsonActivity.get(CALENDAR_EVENT_START_DATE));
                eventInfo.put(CALENDAR_EVENT_END_DATE, jsonActivity.get(CALENDAR_EVENT_END_DATE));
                eventInfo.put(CALENDAR_EVENT_FULL_DATE, jsonActivity.get(CALENDAR_EVENT_FULL_DATE));
                eventInfo.put(CALENDAR_EVENT_RECURRENCE, jsonActivity.get(CALENDAR_EVENT_RECURRENCE));
                eventInfo.put(CALENDAR_EVENT_REPEATER_UNTIL, jsonActivity.get(CALENDAR_EVENT_REPEATER_UNTIL));
                
                mergedEvents.add(eventInfo);
            }
        }
        
        mergedActivities.put("events", mergedEvents);
        mergedActivities.put("amount", mergedEvents.size());
        
        return mergedActivities;
    }
    
    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        
        Calendar calendar = (Calendar) args.get(ObservationConstants.ARGS_CALENDAR);
        
        return getParentProject(calendar);
    }
    
    /**
     * Check if activity is generic type, and should be handled by CalendarEventActivityNotifier, or a more specific notifier
     * @return true if activity is generic type
     */
    public boolean isGenericCalendarEventActivityType()
    {
        return true;
    }
}
