/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.projects;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.cms.repository.comment.CommentableContent;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityFactory;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelLessComposite;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.web.repository.content.WebContent;

/**
 * Activity type to represent that a content was commented
 */
public class ContentCommentedActivityType extends WebContentActivityType
{
    /** the data name for the composite storing the parent comment info */
    public static final String PARENT_COMMENT = "parentComment";
    /** the data path for the parent comment date */
    public static final String COMMENT_DATE = ActivityFactory.DATE;
    /** the data path for the parent comment author */
    public static final String COMMENT_AUTHOR = ActivityFactory.AUTHOR;
    /** the data path for the parent comment summary */
    public static final String PARENT_COMMENT_SUMMARY = "summary";
    /** the data path for the parent comment summary */
    public static final String COMMENT_ID = "commentId";
    /** the data name for the comment */
    public static final String COMMENT_SUMMARY = "commentSummary";

    private PopulationContextHelper _populationContextHelper;
    private ContentTypesHelper _cTypeHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _populationContextHelper = (PopulationContextHelper) serviceManager.lookup(PopulationContextHelper.ROLE);
        _cTypeHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    public boolean support(Event event)
    {
        return _isSupportedContentType(event);
    }
    
    private boolean _isSupportedContentType(Event event)
    {
        // Ignore content that are not wall content or project news
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        if (_cTypeHelper.isInstanceOf(content, WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID)
            || _cTypeHelper.isInstanceOf(content, WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID))
        {
            return true;
        }
        return false;
    }
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        CommentableContent content = (CommentableContent) parameters.get(ObservationConstants.ARGS_CONTENT);
        // We need to store the content author and creation date as it's different from the activity one
        activity.setValue("contentCreator", content.getCreator());
        activity.setValue("contentCreationDate", content.getCreationDate());
        
        Comment comment = (Comment) parameters.get(ObservationConstants.ARGS_COMMENT);
        String id = comment.getId();
        if (comment.isSubComment())
        {
            // We are handling a sub-comment so we want to also store the parent comment information
            Comment parentComment = comment.getCommentParent();
            
            ModifiableModelLessComposite activityParentComment = activity.getComposite(PARENT_COMMENT, true);
            activityParentComment.setValue(COMMENT_SUMMARY, StringUtils.abbreviate(parentComment.getContent(), "…", 150));
            // As with the module, we try to retrive the user by its mail and otherwise, we just provide the fullname
            WebContent web = (WebContent) content;
            String siteName = web.getSite().getName();
            Set<String> userPopulationsOnSite = _populationContextHelper.getUserPopulationsOnContexts(Arrays.asList("/sites/" + siteName, "/sites-fo/" + siteName), false, false);
            try
            {
                User author = _userManager.getUserByEmail(userPopulationsOnSite, parentComment.getAuthorEmail());
                activityParentComment.setValue(COMMENT_AUTHOR, author.getIdentity());
            }
            catch (NotUniqueUserException e)
            {
                activityParentComment.setValue(COMMENT_AUTHOR + "_fullName", parentComment.getAuthorName());
            }
            
            activityParentComment.setValue(COMMENT_DATE, parentComment.getCreationDate());
            activityParentComment.setValue(COMMENT_ID, parentComment.getId());
        }
        
        // finally we store the comment summary
        activity.setValue(COMMENT_ID, id);
        activity.setValue(COMMENT_SUMMARY, StringUtils.abbreviate(comment.getContent(), "…", 150));
    }
}
