/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.tasks;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;

/**
 * {@link ActivityType} implementation for documents-related activities
 */
public class TasksActivityType extends AbstractWorkspacesActivityType
{
    /** Constant for activity's category */
    public static final String ACIVITY_CATEGORY_TASKS = "tasks";
    /** data name for the task title */
    public static final String TASK_TITLE = "taskTitle";
    /** data name for the task id */
    public static final String TASK_ID = "taskId";
    /** data name for the number of attachements of the task */
    public static final String ATTACHEMENTS_COUNT = "nbAttachments";
    
    private TasksWorkspaceModule _taskModule;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _taskModule = moduleManagerEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
    }
    
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        // Category
        activity.setValue(PROJECT_CATEGORY, ACIVITY_CATEGORY_TASKS);

        // Task 
        Task task = (Task) parameters.get(ObservationConstants.ARGS_TASK);
        activity.setValue(TASK_TITLE, task.getLabel());
        activity.setValue(TASK_ID, task.getId());
        activity.setValue(ATTACHEMENTS_COUNT, task.getAttachments().size());
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        
        // add the url if possible
        String projectName = activity.getValue(PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        if (!ObservationConstants.EVENT_TASK_DELETING.equals(activity.getEventType()) && activity.hasValue(TASK_ID))
        {
            String taskId = activity.getValue(TASK_ID);
            json.put("taskUrl", _taskModule.getTaskUri(project, taskId));
        }
        else
        {
            json.put("taskUrl", _taskModule.getModuleUrl(project));
        }
        
        return json;
    }
    
    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivities = super.mergeActivities(activities);
        
        List<Map<String, Object>> tasks = new ArrayList<>();
        
        for (Activity event : activities)
        {
            Map<String, Object> json = event.toJSONForClient();
            Map<String, Object> eventInfo = new HashMap<>();
            
            eventInfo.put(TASK_TITLE, json.get(TASK_TITLE));
            eventInfo.put(TASK_ID, json.get(TASK_ID));
            eventInfo.put("taskUrl", json.get("taskUrl"));
            
            tasks.add(eventInfo);
        }
        
        mergedActivities.put("tasks", tasks);
        
        return mergedActivities;
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        return new StringExpression(TASK_TITLE, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
    }
    
    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        
        Task task = (Task) args.get(ObservationConstants.ARGS_TASK);
        
        return getParentProject(task);
    }
    
    /**
     * Check if activity is generic type, and should be handled by TasksActivityNotifier, or a more specific notifier
     * @return true if activity is generic type
     */
    public boolean isGenericTaskActivityType()
    {
        return true;
    }
}
