/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.calendars.events;

import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.IllegalClassException;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.calendars.AbstractCalendarDAO;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.calendars.ModifiableCalendar;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendar;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEvent;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.workflow.AbstractNodeWorkflowComponent;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;

/**
 * Calendar event DAO
 */
public class CalendarEventDAO extends AbstractCalendarDAO
{

    /** Avalon Role */
    public static final String ROLE = CalendarEventDAO.class.getName();

    /** The tasks list JSON helper */
    protected CalendarEventJSONHelper _calendarEventJSONHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _calendarEventJSONHelper = (CalendarEventJSONHelper) manager.lookup(CalendarEventJSONHelper.ROLE);
    }
    

    /**
     * Get the events between two dates
     * @param startDateAsStr The start date.
     * @param endDateAsStr The end date.
     * @return the events between two dates
     */
    @Callable
    public List<Map<String, Object>> getEvents(String startDateAsStr, String endDateAsStr)
    {
        ZonedDateTime startDate = startDateAsStr != null ? DateUtils.parseZonedDateTime(startDateAsStr) : null;
        ZonedDateTime endDate = endDateAsStr != null ? DateUtils.parseZonedDateTime(endDateAsStr) : null;

        return getEvents(startDate, endDate)
                .stream()
                .map(event -> {
                    Map<String, Object> eventData = _calendarEventJSONHelper.eventAsJson(event, false, false);
                    
                    List<Object> occurrencesDataList = new ArrayList<>();
                    eventData.put("occurrences", occurrencesDataList);
                    
                    List<CalendarEventOccurrence> occurrences = event.getOccurrences(startDate, endDate);
                    for (CalendarEventOccurrence occurrence : occurrences)
                    {
                        occurrencesDataList.add(occurrence.toJSON());
                    }
                    return eventData;
                })
                .collect(Collectors.toList());
    }
    
    /**
     * Get the events between two dates
     * @param startDate Begin date
     * @param endDate End date
     * @return the list of events
     */
    public List<CalendarEvent> getEvents(ZonedDateTime startDate, ZonedDateTime endDate)
    {

        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        Project project = _getProject();
        AmetysObjectIterable<Calendar> calendars = calendarModule.getCalendars(project);

        List<CalendarEvent> eventList = new ArrayList<>();
        if (calendars != null)
        {
            for (Calendar calendar : calendars)
            {
                if (calendarModule.canView(calendar))
                {
                    for (Map.Entry<CalendarEvent, List<CalendarEventOccurrence>> entry : calendar.getEvents(startDate, endDate).entrySet())
                    {
                        CalendarEvent event = entry.getKey();
                        eventList.add(event);
                    }
                }
            }
        }
      
        Calendar resourceCalendar = calendarModule.getResourceCalendar(_getProject());

        for (Map.Entry<CalendarEvent, List<CalendarEventOccurrence>> entry : resourceCalendar.getEvents(startDate, endDate).entrySet())
        {
            CalendarEvent event = entry.getKey();
            eventList.add(event);
        }
        
        return eventList;
    }
    
    /**
     * Delete an event
     * @param id The id of the event
     * @param occurrence a string representing the occurrence date (ISO format).
     * @param choice The type of modification
     * @return The result map with id, parent id and message keys
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    @Callable
    public Map<String, Object> deleteEvent(String id, String occurrence, String choice) throws IllegalAccessException
    {
        if (!"unit".equals(choice))
        {
            JCRCalendarEvent event = _resolver.resolveById(id);
            _messagingConnectorCalendarManager.deleteEvent(event);
        }
        
        Map<String, Object> result = new HashMap<>();

        assert id != null;
        
        AmetysObject object = _resolver.resolveById(id);
        if (!(object instanceof ModifiableCalendarEvent))
        {
            throw new IllegalClassException(ModifiableCalendarEvent.class, object.getClass());
        }
        
        ModifiableCalendarEvent event = (ModifiableCalendarEvent) object;
        ModifiableCalendar calendar = event.getParent();
        
        // Check user right
        try
        {
            _explorerResourcesDAO.checkUserRight(calendar, RIGHTS_EVENT_DELETE);
        }
        catch (IllegalAccessException e)
        {
            UserIdentity user = _currentUserProvider.getUser();
            UserIdentity creator = event.getCreator();
            RightResult rightCreator = _rightManager.hasRight(user, RIGHTS_EVENT_DELETE_OWN, calendar);
            boolean hasOwnDeleteRight = rightCreator == RightResult.RIGHT_ALLOW && creator.equals(user);
            if (!hasOwnDeleteRight)
            {
                // rethrow exception
                throw e;
            }
        }
        
        if (!_explorerResourcesDAO.checkLock(event))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to delete event'" + object.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        String parentId = calendar.getId();
        String name = event.getName();
        String path = event.getPath();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.plugins.workspaces.calendars.ObservationConstants.ARGS_CALENDAR, calendar);
        eventParams.put(ObservationConstants.ARGS_ID, id);
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, path);
        eventParams.put(org.ametys.plugins.workspaces.calendars.ObservationConstants.ARGS_CALENDAR_EVENT, event);
       
        if (StringUtils.isNotBlank(choice) && choice.equals("unit"))
        {
            ArrayList<ZonedDateTime> excludedOccurrences = new ArrayList<>();
            excludedOccurrences.addAll(event.getExcludedOccurences());
            ZonedDateTime occurrenceDate = DateUtils.parseZonedDateTime(occurrence).withZoneSameInstant(event.getZone());
            excludedOccurrences.add(occurrenceDate.truncatedTo(ChronoUnit.DAYS));

            _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_DELETING, _currentUserProvider.getUser(), eventParams));
            
            event.setExcludedOccurrences(excludedOccurrences);
        }
        else
        {
            _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_DELETING, _currentUserProvider.getUser(), eventParams));
            
            event.remove();
        }
        
        calendar.saveChanges();
        
        result.put("id", id);
        result.put("parentId", parentId);
        
        eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, id);
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_DELETED, _currentUserProvider.getUser(), eventParams));
        
        return result;
    }

    /**
     * Add an event and return it. Use the calendar view dates to compute occurrences between those dates.
     * @param parameters The map of parameters to perform the action
     * @param calendarViewStartDateAsStr The calendar view start date, compute occurrences after this date.
     * @param calendarViewEndDateAsStr The calendar view end date, compute occurrences before this date.
     * @return The map of results populated by the underlying workflow action
     * @throws WorkflowException if an error occurred
     * @throws IllegalAccessException  If the user does not have the right to add an event
     */
    @Callable
    public Map<String, Object> addEvent(Map<String, Object> parameters, String calendarViewStartDateAsStr, String calendarViewEndDateAsStr) throws WorkflowException, IllegalAccessException
    {
        ZonedDateTime calendarViewStartDate = calendarViewStartDateAsStr != null ? DateUtils.parseZonedDateTime(calendarViewStartDateAsStr) : null;
        ZonedDateTime calendarViewEndDate = calendarViewEndDateAsStr != null ? DateUtils.parseZonedDateTime(calendarViewEndDateAsStr) : null;
        
        Map<String, Object> result = doWorkflowEventAction(parameters);
        
        //TODO Move to create event action (workflow) ?
        String eventId = (String) result.get("id");
        _messagingConnectorCalendarManager.addEventInvitation(parameters, eventId);
        
        _projectManager.getProjectsRoot().saveChanges();

        JCRCalendarEvent event = _resolver.resolveById((String) result.get("id"));
        Map<String, Object> eventDataWithFilteredOccurences = _calendarEventJSONHelper.eventAsJsonWithOccurrences(event, false, calendarViewStartDate, calendarViewEndDate);
        
        result.put("eventDataWithFilteredOccurences", eventDataWithFilteredOccurences);
        
        return result;
    }
    
    /**
     * Edit an event
     * @param parameters The map of parameters to perform the action
     * @param calendarViewStartDateAsStr The calendar view start date, compute occurrences after this date.
     * @param calendarViewEndDateAsStr The calendar view end date, compute occurrences before this date.
     * @return The map of results populated by the underlying workflow action
     * @throws WorkflowException if an error occurred
     */
    @Callable
    public Map<String, Object> editEvent(Map<String, Object> parameters, String calendarViewStartDateAsStr, String calendarViewEndDateAsStr) throws WorkflowException
    {
        ZonedDateTime calendarViewStartDate = calendarViewStartDateAsStr != null ? DateUtils.parseZonedDateTime(calendarViewStartDateAsStr) : null;
        ZonedDateTime calendarViewEndDate = calendarViewEndDateAsStr != null ? DateUtils.parseZonedDateTime(calendarViewEndDateAsStr) : null;
        
        String eventId = (String) parameters.get("id");
        JCRCalendarEvent event = _resolver.resolveById(eventId);
        
        // handle event move if calendar has changed
        String previousCalendarId = event.getParent().getId();
        String parentId = (String) parameters.get("parentId");
        
        if (previousCalendarId != null && !StringUtils.equals(parentId, previousCalendarId))
        {
            JCRCalendar parentCalendar = _resolver.resolveById(parentId);
            move(event, parentCalendar);
        }
        
        Map<String, Object> result = doWorkflowEventAction(parameters);
        
        //TODO Move to edit event action (workflow) ?
        String choice = (String) parameters.get("choice");
        if (!"unit".equals(choice))
        {
            _messagingConnectorCalendarManager.editEventInvitation(parameters, eventId);
        }
        
        _projectManager.getProjectsRoot().saveChanges();

        Map<String, Object> oldEventData = _calendarEventJSONHelper.eventAsJsonWithOccurrences(event, false, calendarViewStartDate, calendarViewEndDate);
        JCRCalendarEvent newEvent = _resolver.resolveById((String) result.get("id"));
        Map<String, Object> newEventData = _calendarEventJSONHelper.eventAsJsonWithOccurrences(newEvent, false, calendarViewStartDate, calendarViewEndDate);
        
        result.put("oldEventData", oldEventData);
        result.put("newEventData", newEventData);
        
        return result;
    }
    
    /**
     * Move a event to another calendar
     * @param event The event to move
     * @param parent The new parent calendar
     * @throws AmetysRepositoryException if an error occurred while moving
     */
    public void move(JCRCalendarEvent event, JCRCalendar parent) throws AmetysRepositoryException
    {
        try
        {
            event.getNode().getSession().move(event.getNode().getPath(), parent.getNode().getPath() + "/ametys:calendar-event");
            
            Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(event);

            String previousWorkflowName = workflow.getWorkflowName(event.getWorkflowId());
            String workflowName = parent.getWorkflowName();

            if (!StringUtils.equals(previousWorkflowName, workflowName))
            {
                // If both calendar have a different workflow, initialize a new workflow instance for the event
                HashMap<String, Object> inputs = new HashMap<>();
                inputs.put(AbstractNodeWorkflowComponent.EXPLORERNODE_KEY, parent);
                workflow = _workflowProvider.getAmetysObjectWorkflow(event);
                
                long workflowId = workflow.initialize(workflowName, 0, inputs);
                event.setWorkflowId(workflowId);
            }
        }
        catch (WorkflowException | RepositoryException e)
        {
            String errorMsg = String.format("Fail to move the event '%s' to the calendar '%s'.", event.getId(), parent.getId());
            throw new AmetysRepositoryException(errorMsg, e);
        }
    }
    
    /**
     * Do an event workflow action
     * @param parameters The map of action parameters
     * @return The map of results populated by the workflow action
     * @throws WorkflowException if an error occurred
     */
    @Callable
    public Map<String, Object> doWorkflowEventAction(Map<String, Object> parameters) throws WorkflowException
    {
        Map<String, Object> result = new HashMap<>();
        HashMap<String, Object> inputs = new HashMap<>();

        inputs.put("parameters", parameters);
        inputs.put("result", result);
        
        String eventId = (String) parameters.get("id");
        Long workflowInstanceId = null;
        CalendarEvent event = null;
        if (StringUtils.isNotEmpty(eventId))
        {
            event = _resolver.resolveById(eventId);
            workflowInstanceId = event.getWorkflowId();
        }
        
        inputs.put("eventId", eventId);
        
        Calendar calendar = null;
        String calendarId = (String) parameters.get("parentId");
        
        if (StringUtils.isNotEmpty(calendarId))
        {
            calendar = _resolver.resolveById(calendarId);
        }
        // parentId can be not provided for some basic actions where the event already exists
        else if (event != null)
        {
            calendar = event.getParent();
        }
        else
        {
            throw new WorkflowException("Unable to retrieve the current calendar");
        }
        
        inputs.put(AbstractNodeWorkflowComponent.EXPLORERNODE_KEY, calendar);
        
        String workflowName = calendar.getWorkflowName();
        if (workflowName == null)
        {
            throw new IllegalArgumentException("The workflow name is not specified");
        }
        
        int actionId =  (int) parameters.get("actionId");
        
        boolean sendMail = true;
        String choice = (String) parameters.get("choice");
        if (actionId == 2 && "unit".equals(choice))
        {
            sendMail = false;
        }
        inputs.put("sendMail", sendMail);
        
        Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(event != null ? event : null);
        
        if (workflowInstanceId == null)
        {
            try
            {
                workflow.initialize(workflowName, actionId, inputs);
            }
            catch (WorkflowException e)
            {
                getLogger().error("An error occured while creating workflow '" + workflowName + "' with action '" + actionId, e);
                throw e;
            }
        }
        else
        {
            try
            {
                workflow.doAction(workflowInstanceId, actionId, inputs);
            }
            catch (WorkflowException e)
            {
                getLogger().error("An error occured while doing action '" + actionId + "'with the workflow '" + workflowName, e);
                throw e;
            }
        }
        
        return result;
    }
    
}
