/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.catalog;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.observation.Observer;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.initialization.CatalogPageInitializerExtensionPoint;
import org.ametys.plugins.workspaces.initialization.PageInitializer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Observer that initialize a new catalog site.
 * This observer will create the different page based on models configuration,
 * adding the usual service and content required for proper usage of a catalog.
 * The observer will also set basic right configuration.
 */
public class InitializeCatalogSitemapObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    private ObservationManager _observationManager;
    private ProfileAssignmentStorageExtensionPoint _profileAssignementStorageEP;
    private CurrentUserProvider _currentUserProvider;
    private CatalogPageInitializerExtensionPoint _catalogPageInitializerEP;

    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _catalogPageInitializerEP = (CatalogPageInitializerExtensionPoint) manager.lookup(CatalogPageInitializerExtensionPoint.ROLE);
        _profileAssignementStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
    }
    
    public boolean supports(Event event)
    {
        // Listen to updated event as they are the one creating the sitemaps
        if (ObservationConstants.EVENT_SITE_UPDATED.equals(event.getId()))
        {
            Site site = (Site) event.getArguments().get(ObservationConstants.ARGS_SITE);
            return CatalogSiteType.TYPE_ID.equals(site.getType());
        }
        return false;
    }

    public int getPriority(Event event)
    {
        // Wait for live synchronization by {@link AbstractSynchronizeObserver}
        return MAX_PRIORITY + 2000;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Site site = (Site) event.getArguments().get(ObservationConstants.ARGS_SITE);
        initializeSitemaps(site);
    }

    private void initializeSitemaps(Site site)
    {
        try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
        {
            for (Sitemap sitemap : sitemaps)
            {
                try (AmetysObjectIterable< ? extends Page> pages = sitemap.getChildrenPages())
                {
                    // Only execute on fresh sitemap
                    if (pages.getSize() == 0)
                    {
                        _setRightsOnContext(sitemap);
                        
                        for (String initializerId : _catalogPageInitializerEP.getExtensionsIds())
                        {
                            PageInitializer initializer = _catalogPageInitializerEP.getExtension(initializerId);
                            initializer.createPage(sitemap);
                        }
                        
                        Map<String, Object> eventParams = new HashMap<>();
                        eventParams.put(ObservationConstants.ARGS_SITEMAP, sitemap);
                        _observationManager.notify(new Event(ObservationConstants.EVENT_SITEMAP_UPDATED, _currentUserProvider.getUser(), eventParams));
                    }
                }
            }
        }
    }

    private void _setRightsOnContext(Sitemap sitemap)
    {
        _profileAssignementStorageEP.denyProfileToAnonymous(RightManager.READER_PROFILE_ID, sitemap);
        _profileAssignementStorageEP.allowProfileToAnyConnectedUser(RightManager.READER_PROFILE_ID, sitemap);
        sitemap.saveChanges();
        
        _notifyACLChange(sitemap, Set.of(RightManager.READER_PROFILE_ID));
    }
    
    private void _notifyACLChange(Object context, Set<String> profilesId)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, context);
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, profilesId);
        
        _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
}
