/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.chat;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;

/**
 * Job to ensure that chat server is synchronized with ametys: users and chatrooms. 
 */
public class ServersSynchronizationSchedulable extends AbstractStaticSchedulable
{
    private ChatHelper _chatHelper;
    private ProjectManager _projectManager;
    private ProjectMemberManager _projectMemberManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _chatHelper = (ChatHelper) manager.lookup(ChatHelper.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        if (!Config.getInstance().<Boolean>getValue("workspaces.chat.active"))
        {
            return;
        }
        
        ContextHelper.getRequest(_context).setAttribute("site", Config.getInstance().getValue("workspaces.catalog.site.name"));
        
        int errors = 0;
        
        // The set of users already updated to ensure to update only once each user
        Set<UserIdentity> handledUsers = new HashSet<>();
        
        for (String projectName : _projectManager.getProjectNames())
        {
            Project project = _projectManager.getProject(projectName);
            
            try
            {
                // Ensure that chatroom exists
                _chatHelper.getRoom(projectName, true);
                
                // Prepare correct user lis
                Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, true);
                List<UserIdentity> projectMemersIdentities = projectMembers.stream().map(pm -> pm.getUser().getIdentity()).collect(Collectors.toList());
                
                // Remove any user in the chatroom
                _chatHelper.removeAllUsersFromRoom(projectName, projectMemersIdentities);
                
                for (UserIdentity userIdentity : projectMemersIdentities)
                {
                    try
                    {
                        if (!handledUsers.contains(userIdentity))
                        {
                            // Ensure user exists
                            _chatHelper.getUser(userIdentity, true);
                            
                            // Update existing users
                            _chatHelper.updateUserInfos(userIdentity, false);
                            
                            handledUsers.add(userIdentity);
                        }
                        
                        // Add user to the chat room
                        _chatHelper.addUserToRoom(userIdentity, projectName);
                    }
                    catch (Exception e)
                    {
                        getLogger().error("An error occurred when syncronizing user " + UserIdentity.userIdentityToString(userIdentity) + " with project " + projectName, e);
                        errors++;
                    }
                }
            }
            catch (Exception e)
            {
                getLogger().error("An error occurred when syncronizing project " + projectName, e);
                errors++;
            }
        }
        
        if (errors > 0)
        {
            throw new IllegalStateException(errors + " errors occurred while synchronizing chat server. See previous log to know more");
        }
    }

}
