/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.IllegalClassException;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.AbstractWorkspaceDAO;
import org.ametys.plugins.workspaces.forum.jcr.JCRThread;
import org.ametys.plugins.workspaces.forum.json.ThreadJSONHelper;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Abstract class for forum module DAO's
 */
public abstract class AbstractWorkspaceThreadDAO extends AbstractWorkspaceDAO
{
    
    /** The project tags DAO */
    protected ThreadJSONHelper _threadJSONHelper;

    /** The workspace thread user preferences DAO */
    protected WorkspaceThreadUserPreferencesDAO _workspaceThreadUserPrefDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _threadJSONHelper = (ThreadJSONHelper) manager.lookup(ThreadJSONHelper.ROLE);
        _workspaceThreadUserPrefDAO = (WorkspaceThreadUserPreferencesDAO) manager.lookup(WorkspaceThreadUserPreferencesDAO.ROLE);
    }
    
    /**
     * Get the thread root
     * @param projectName the project name
     * @return the thread root
     */
    protected ModifiableTraversableAmetysObject _getThreadRoot(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        
        ForumWorkspaceModule threadModule = _workspaceModuleEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
        return threadModule.getModuleRoot(project, true);
    }

    /**
     * Get a thread by id
     * @param threadId the thread ID
     * @return the thread, or an IllegalClassException if the object is not a Thread
     */
    protected JCRThread _getThreadByID(String threadId)
    {
        AmetysObject object = _resolver.resolveById(threadId);
        
        if (!(object instanceof JCRThread))
        {
            throw new IllegalClassException(JCRThread.class, object.getClass());
        }

        JCRThread thread = (JCRThread) object;
        return thread;
    }

    /**
     * Get the module root of thread
     * @return the module root
     */
    protected ModifiableResourceCollection _getModuleRoot()
    {
        String projectName = _getProjectName();
        return _getModuleRoot(projectName);
    }
    
    /**
     * Get the module root of thread
     * @param projectName the project name
     * @return the module root
     */
    protected ModifiableResourceCollection _getModuleRoot(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        ForumWorkspaceModule forumModule = _workspaceModuleEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
        return forumModule.getModuleRoot(project, false);
    }

    /**
     * Check user rights : the user can either have a global right on the root, or a specific right on the root while being the author of the thread
     * @param objectToCheck the object to check
     * @param superRightId the right id
     * @param thread the thread
     * @param authorRightId the right to check as we check if the current user is the author of thread
     * @throws IllegalAccessException if a right error occurred
     */
    protected void _checkUserRights(AmetysObject objectToCheck, String superRightId, Thread thread, String authorRightId) throws IllegalAccessException
    {
        UserIdentity user = _currentUserProvider.getUser();
        if (_rightManager.hasRight(user, superRightId, objectToCheck) == RightResult.RIGHT_ALLOW)
        {
            return;
        }
        if (_rightManager.hasRight(user, authorRightId, objectToCheck) == RightResult.RIGHT_ALLOW && thread.getAuthor().equals(user))
        {
            return;
        }
        throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to do operation without convenient right [" + superRightId + "]");
    }
}
