/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.languages.Language;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.workspaces.members.MembersWorkspaceModule.Invitation;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.usermanagement.UserManagementException;
import org.ametys.web.usermanagement.UserManagementException.StatusError;
import org.ametys.web.usermanagement.UserSignUpConfiguration;
import org.ametys.web.usermanagement.UserSignupManager;

import jakarta.mail.MessagingException;

/**
 * Helper for invitations by email
 *
 */
public class ProjectInvitationHelper extends AbstractLogEnabled implements Serviceable, Component, Configurable, PluginAware, Contextualizable
{
    /** The role */
    public static final String ROLE = ProjectInvitationHelper.class.getName();
    
    private static final String __MAIL_PROJECT_EMAIL_PATTERN = "${email}";
    private static final String __MAIL_PROJECT_TOKEN_PATTERN = "${token}";
    
    private ProjectManager _projectManager;
    private UserSignUpConfiguration _signupConfig;
    private UserSignupManager _signupManager;
    private SiteManager _siteManager;
    private WorkspaceModuleExtensionPoint _moduleEP;
    private CurrentUserProvider _currentUserProvider;
    private UserManager _userManager;
    private UserHelper _userHelper;
    private ProjectRightHelper _projectRightsHelper;
    private ProjectMemberManager _projectMemberManager;
    private PopulationContextHelper _populationContextHelper;
    private LanguagesManager _languagesManager;
    
    private String _subjectKeyForInvitation;
    private String _textBodyKeyForInvitation;
    private String _htmlBodyKeyForInvitation;
    private String _subjectKeyForInvitationAccepted;
    private String _textBodyKeyForInvitationAccepted;
    private String _htmlBodyKeyForInvitationAccepted;

    private I18nUtils _i18nUtils;

    private String _pluginName;

    private Context _context;


    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _projectRightsHelper = (ProjectRightHelper) serviceManager.lookup(ProjectRightHelper.ROLE);
        _projectMemberManager = (ProjectMemberManager) serviceManager.lookup(ProjectMemberManager.ROLE);
        _signupConfig = (UserSignUpConfiguration) serviceManager.lookup(UserSignUpConfiguration.ROLE);
        _signupManager = (UserSignupManager) serviceManager.lookup(UserSignupManager.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _populationContextHelper = (PopulationContextHelper) serviceManager.lookup(PopulationContextHelper.ROLE);
        _languagesManager = (LanguagesManager) serviceManager.lookup(LanguagesManager.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _subjectKeyForInvitation = configuration.getChild("invitation-email-subject").getValue(null);
        _textBodyKeyForInvitation = configuration.getChild("invitation-email-text-body").getValue(null);
        _htmlBodyKeyForInvitation = configuration.getChild("invitation-email-html-body").getValue(null);
        
        _subjectKeyForInvitationAccepted = configuration.getChild("invitation-accepted-email-subject").getValue(null);
        _textBodyKeyForInvitationAccepted = configuration.getChild("invitation-accepted-email-text-body").getValue(null);
        _htmlBodyKeyForInvitationAccepted = configuration.getChild("invitation-accepted-email-html-body").getValue(null);
    }
    
    /**
     * Invite emails to be member of a project
     * @param projectName The project name
     * @param emails The emails
     * @param allowedProfileByModule the allowed profiles by module
     * @return The result
     * @throws UserManagementException if failed to invit user
     * @throws NotUniqueUserException if many users match the given email
     */
    public Map<String, Object> inviteEmails(String projectName, List<String> emails, Map<String, String> allowedProfileByModule) throws UserManagementException, NotUniqueUserException
    {
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request);
        String sitemapLanguage = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
        
        return inviteEmails(projectName, siteName, sitemapLanguage, emails, allowedProfileByModule);
    }
    /**
     * Invite emails to be member of a project
     * @param projectName The project name
     * @param siteName The site name
     * @param lang the current language
     * @param emails The emails
     * @param allowedProfileByModule the allowed profiles by module
     * @return The result
     * @throws UserManagementException if failed to invit user
     * @throws NotUniqueUserException if many users match the given email
     */
    @Callable
    public Map<String, Object> inviteEmails(String projectName, String siteName, String lang, List<String> emails, Map<String, String> allowedProfileByModule) throws UserManagementException, NotUniqueUserException
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("existing-users", new ArrayList<>());
        result.put("email-success", new ArrayList<>());
        result.put("email-error", new ArrayList<>());
        
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            MembersWorkspaceModule module = _moduleEP.getModule(MembersWorkspaceModule.MEMBERS_MODULE_ID);
            if (module != null && _projectManager.isModuleActivated(project, module.getId()))
            {
                if (!_projectRightsHelper.canAddMember(project))
                {
                    throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to send invitations without sufficient right."); 
                }
                
                String catalogSiteName = _projectManager.getCatalogSiteName();
                
                UserDirectory userDirectory = _getUserDirectoryForSignup(result, catalogSiteName, lang);
                if (userDirectory != null)
                {
                    String populationId = userDirectory.getPopulationId();
                    String userDirectoryId = userDirectory.getId();
                    
                    // Prepare mail
                    Map<String, String> emailConfiguration = _getInvitationEmailConfiguration(project, lang);
                    String mailSubject = emailConfiguration.get("subject");
                    String mailBodyText = emailConfiguration.get("bodyText");
                    String mailBodyHtml = emailConfiguration.get("bodyHtml");
                    
                    for (String email : emails)
                    {
                        try
                        {
                            _signupManager.inviteToSignup(catalogSiteName, lang, email, populationId, userDirectoryId, null, null, false, false, false);
                            
                            if (_addOrUpdateInvitation(project, catalogSiteName, module, email, allowedProfileByModule, populationId, userDirectoryId, mailSubject, mailBodyText, mailBodyHtml))
                            {
                                @SuppressWarnings("unchecked")
                                List<String> emailSuccess = (List<String>) result.get("email-success");
                                emailSuccess.add(email);
                                
                            }
                        }
                        catch (UserManagementException e)
                        {
                            switch (e.getStatusError())
                            {
                                case USER_ALREADY_EXISTS:
                                    User user = _getUser(catalogSiteName, email);
                                    
                                    Map<String, Object> user2json = _userHelper.user2json(user, true);
                                    
                                    @SuppressWarnings("unchecked")
                                    List<Map<String, Object>> existingUsers = (List<Map<String, Object>>) result.get("existing-users");
                                    existingUsers.add(user2json);
                                    break;
                                    
                                case TEMP_USER_ALREADY_EXISTS:
                                    // if mail has already been invited, re-created the invitation
                                    if (_addOrUpdateInvitation(project, catalogSiteName, module, email, allowedProfileByModule, populationId, userDirectoryId, mailSubject, mailBodyText, mailBodyHtml))
                                    {
                                        @SuppressWarnings("unchecked")
                                        List<String> emailSuccess = (List<String>) result.get("email-success");
                                        emailSuccess.add(email);
                                        
                                    }
                                    break;
                                    
                                default:
                                    getLogger().error("Cannot invite " + email, e);
                                    @SuppressWarnings("unchecked")
                                    List<String> emailErrors = (List<String>) result.get("email-error");
                                    emailErrors.add(email);
                                    break;
                            }
                        }
                    }
                    
                    @SuppressWarnings("unchecked")
                    List<String> emailSuccess = (List<String>) result.get("email-success");
                    if (emailSuccess.size() == emails.size())
                    {
                        result.put("success", true);
                    }
                }
            }
        }
        else
        {
            result.put("success", false);
            result.put("unknown-project", projectName);
        }
        
        return result;
    }
    
    private User _getUser(String siteName, String email) throws NotUniqueUserException
    {
        Set<String> populations = _populationContextHelper.getUserPopulationsOnContexts(Arrays.asList("/sites/" + siteName, "/sites-fo/" + siteName), false);
        for (String population : populations)
        {
            User user = _userManager.getUser(population, email);
            if (user == null)
            {
                user = _userManager.getUserByEmail(population, email);
            }
            
            if (user != null)
            {
                return user;
            }
        }
        
        return null;
    }
    
    /**
     * Get the configuration to invite users by emails
     * @param projectName The current project
     * @param lang the current language
     * @return the configuration for email invitations
     */
    @Callable
    public Map<String, Object> getInvitationConfiguration(String projectName, String lang)
    {
        Project project = _projectManager.getProject(projectName);
        
        Map<String, Object> config = new HashMap<>();
        
        // Check the configuration is valid for invitations
        String catalogSiteName = _projectManager.getCatalogSiteName();
        if (_getUserDirectoryForSignup(config, catalogSiteName, lang) == null)
        {
            config.remove("success");
            config.put("allowed", false); 
            return config;
        }
        
        // Check the current user has right to invite users
        if (!_projectRightsHelper.canAddMember(project))
        {
            config.put("allowed", false); 
            config.put("error", "no-right"); 
            return config;
        }
        
        config.put("allowed", true);
        config.put("email", _getInvitationEmailConfiguration(project, lang));
        config.put("rights", _projectRightsHelper.getProjectRightsData(projectName)); 
        
        return config;
    }
    
    private Map<String, String> _getInvitationEmailConfiguration(Project project, String lang)
    {
        Map<String, String> emailConfig = new HashMap<>();
        
        Map<String, I18nizableTextParameter> i18nparams = new HashMap<>();
        i18nparams.put("projectTitle", new I18nizableText(project.getTitle()));
        i18nparams.put("projectUrl", new I18nizableText(_projectManager.getProjectUrl(project, StringUtils.EMPTY)));
        i18nparams.put("nbDays", new I18nizableText(String.valueOf(_signupConfig.getTokenValidity())));
        i18nparams.put("nbDays", new I18nizableText(String.valueOf(_signupConfig.getTokenValidity())));
        
        String catalogSiteName = _projectManager.getCatalogSiteName();
        Site catalogSite = _siteManager.getSite(catalogSiteName);
        i18nparams.put("catalogSiteTitle", new I18nizableText(catalogSite.getTitle()));
        i18nparams.put("catalogSiteUrl", new I18nizableText(catalogSite.getUrl()));
        
        Page signupPage = _getSignupPage(_projectManager.getCatalogSiteName(), lang);
        if (signupPage != null)
        {
            String signupUri = ResolveURIComponent.resolve("page", signupPage.getId(), false, true) + "?email=${email}&token=${token}";
            i18nparams.put("signupUri", new I18nizableText(signupUri));
        }
        
        String subject = getSubjectForInvitationEmail(i18nparams, lang);
        if (subject != null)
        {
            emailConfig.put("subject", subject);
        }
        String bodyTxt = getTextBodyForInvitationEmail(i18nparams, lang);
        if (bodyTxt != null)
        {
            emailConfig.put("bodyText", bodyTxt);
        }
        String bodyHtml = getHtmlBodyForInvitationEmail(i18nparams, lang);
        if (bodyHtml != null)
        {
            emailConfig.put("bodyHtml", bodyHtml);
        }
        
        return emailConfig;
    }

    private Page _getSignupPage(String catalogSiteName, String lang)
    {
        Page signupPage = _signupManager.getSignupPage(catalogSiteName, lang);

        if (signupPage == null)
        {
            signupPage = _signupManager.getSignupPage(catalogSiteName, "en");
        }
        
        if (signupPage == null)
        {
            Map<String, Language> availableLanguages = _languagesManager.getAvailableLanguages();
            for (Language availableLanguage : availableLanguages.values())
            {
                if (signupPage == null)
                {
                    signupPage = _signupManager.getSignupPage(catalogSiteName, availableLanguage.getCode());
                }
            }
        }
        return signupPage;
    }
    
    private void _sendInvitationMail (String catalogSiteName, String email, String populationId, String userDirectoryId, String mailSubject, String mailBodyText, String mailBodyHtml) throws UserManagementException
    {
        String encodedEmail;
        try
        {
            encodedEmail = URLEncoder.encode(email, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            // Should never happen.
            throw new UserManagementException("Encoding error while sending a sign-up confirmation e-mail.", StatusError.MAIL_ERROR, e);
        }

        String token = _signupManager.getToken(catalogSiteName, email, populationId, userDirectoryId);
        
        String bodyTxt = StringUtils.replace(mailBodyText, __MAIL_PROJECT_TOKEN_PATTERN, token);
        bodyTxt = StringUtils.replace(bodyTxt, __MAIL_PROJECT_EMAIL_PATTERN, encodedEmail);
        
        String bodyHtml = StringUtils.replace(mailBodyHtml, __MAIL_PROJECT_TOKEN_PATTERN, token);
        bodyHtml = StringUtils.replace(bodyHtml, __MAIL_PROJECT_EMAIL_PATTERN, encodedEmail);
        
        getLogger().debug("Sending signup invitation e-mail to {}", email);
        
        Site site = _siteManager.getSite(catalogSiteName);
        String from = site.getValue("site-mail-from");

        try
        {
            List<String> errorReport = new ArrayList<>();
            
            // Send the e-mail.
            SendMailHelper.newMail()
                          .withSubject(mailSubject)
                          .withTextBody(bodyTxt)
                          .withHTMLBody(bodyHtml)
                          .withSender(from)
                          .withRecipient(email)
                          .withErrorReport(errorReport)
                          .sendMail();
            
            if (errorReport.contains(email))
            {
                throw new UserManagementException("Error sending the sign-up confirmation mail.", StatusError.MAIL_ERROR);
            }
        }
        catch (MessagingException | IOException e)
        {
            throw new UserManagementException("Error sending the sign-up confirmation mail.", StatusError.MAIL_ERROR, e);
        }
    }
    
    private boolean _addOrUpdateInvitation(Project project, String catalogSiteName, MembersWorkspaceModule module, String email, Map<String, String> allowedProfileByModules, String populationId, String userDirectoryId, String mailSubject, String mailBodyText, String mailBodyHtml) throws UserManagementException
    {
        try
        {
            // First remove invitation if exists
            module.removeInvitation(project, email);
            
            // Add invitations with temporary rights
            module.addInvitation(project, new Date(), email, _currentUserProvider.getUser(), allowedProfileByModules);
            
            project.saveChanges();

            _sendInvitationMail(catalogSiteName, email, populationId, userDirectoryId, mailSubject, mailBodyText, mailBodyHtml);
            
            return true;
        }
        catch (RepositoryException e)
        {
            getLogger().error("Fail to store invitation for email " + email, e);
            return false;
        }
    }
    
    private UserDirectory _getUserDirectoryForSignup(Map<String, Object> result, String catalogSiteName, String lang)
    {
        if (catalogSiteName == null)
        {
            getLogger().error("The catalog's site name is not configured. User invitations can not be activated.");
            result.put("success", false);
            result.put("error", "invalid-configuration");
            return null;
        }
        
        if (!_signupManager.isSignupAllowed(catalogSiteName))
        {
            getLogger().warn("Signup is disabled for the catalog's site.");
            result.put("success", false);
            return null;
        }
        
        Page signupPage = _getSignupPage(catalogSiteName, lang);
        if (signupPage == null)
        {
            getLogger().error("The catalog's site does not contain the signup service for language " + lang + ". User invitations can not be activated.");
            result.put("success", false);
            result.put("error", "invalid-configuration");
            return null;
        }
        
        UserDirectory userDirectory = _getUserDirectoryForSignup(signupPage);
        if (userDirectory == null)
        {
            getLogger().error("There is no user directory configured for users signup. Please check the sign up service of catalog's site.");
            result.put("success", false);
            result.put("error", "invalid-configuration");
            return null;
        }
        
        return userDirectory;
            
    }
    
    private UserDirectory _getUserDirectoryForSignup(Page signupPage)
    {
        for (Zone zone : signupPage.getZones())
        {
            try (AmetysObjectIterable<? extends ZoneItem> zoneItems = zone.getZoneItems())
            {
                for (ZoneItem zoneItem : zoneItems)
                {
                    UserDirectory userDirectory = _signupManager.getUserDirectory(zoneItem);
                    if (userDirectory != null)
                    {
                        return userDirectory;
                    }
                }
            }
        }
        
        return null;
        
    }
    
    /**
     * Add user as member of all project where it has been invited
     * @param user the new user
     */
    public void createMemberFromInvitations(User user)
    {
        Request request = ContextHelper.getRequest(_context);
        String currentWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, "default");
            
            MembersWorkspaceModule memberModule = (MembersWorkspaceModule) _moduleEP.getExtension(MembersWorkspaceModule.MEMBERS_MODULE_ID);
            
            List<Invitation> invitations = memberModule.getInvitations(user.getEmail());
            
            for (Invitation invitation : invitations)
            {
                Map<String, String> allowedProfileByModules = invitation.getAllowedProfileByModules();
                String projectName = invitation.getProjectName();
                
                Project project = _projectManager.getProject(projectName);
                
                _projectMemberManager.addOrUpdateProjectMember(project, user.getIdentity(), allowedProfileByModules, invitation.getAuthor());
                
                // Notify author that invitation was accepted
                UserIdentity author = invitation.getAuthor();
                sendInvitationAcceptedMail(project, _userManager.getUser(author), user);
                
                try
                {
                    // Remove invitation
                    memberModule.removeInvitation(project, invitation.getEmail());
                }
                catch (RepositoryException e)
                {
                    getLogger().error("Failed to remove invitation " + invitation, e);
                }
            }
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWorkspace);
        }
    
    }
    
    /**
     * Send email to the user who initiated the invitation
     * @param project The project
     * @param invitAuthor The author of invitation
     * @param newMember The new member
     */
    protected void sendInvitationAcceptedMail(Project project, User invitAuthor, User newMember)
    {
        if (invitAuthor != null)
        {
            Site site = project.getSite();
            String from = site.getValue("site-mail-from");
            String email = invitAuthor.getEmail();
            
            if (StringUtils.isNotEmpty(email))
            {
                // Prepare mail.
                Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                i18nParams.put("projectTitle", new I18nizableText(project.getTitle()));
                i18nParams.put("projectUrl", new I18nizableText(_projectManager.getProjectUrl(project, StringUtils.EMPTY)));
                i18nParams.put("newMember", new I18nizableText(newMember.getFullName()));
                i18nParams.put("newMemberMail", new I18nizableText(newMember.getEmail()));
                
                Set<Page> memberPages = _projectManager.getModulePages(project, MembersWorkspaceModule.MEMBERS_MODULE_ID);
                if (!memberPages.isEmpty())
                {
                    Page page = memberPages.iterator().next();
                    i18nParams.put("membersPageUri", new I18nizableText(ResolveURIComponent.resolve("page", page.getId(), false, true)));
                }

                String subject = getSubjectForInvitationAcceptedEmail(i18nParams, null);
                String textBody = getTextBodyForInvitationAcceptedEmail(i18nParams, null);
                String htmlBody = getHtmlBodyForInvitationAcceptedEmail(project, i18nParams, null);

                try
                {
                    // Send the e-mail.
                    SendMailHelper.newMail()
                                  .withSubject(subject)
                                  .withHTMLBody(htmlBody)
                                  .withTextBody(textBody)
                                  .withSender(from)
                                  .withRecipient(email)
                                  .sendMail();
                }
                catch (MessagingException | IOException e)
                {
                    getLogger().error("Error sending the invitation accepted email.", e);
                }
            }
        }
    }
    
    /**
     * The email subject for invitation by email
     * @param defaultI18nParams The default i18n parameters
     * @param language the language
     * @return the email subject for invitation by email
     */
    public String getSubjectForInvitationEmail (Map<String, I18nizableTextParameter> defaultI18nParams , String language)
    {
        if (StringUtils.isNotBlank(_subjectKeyForInvitation))
        {
            return _i18nUtils.translate(_getI18nizableText(_subjectKeyForInvitation, defaultI18nParams), language);
        }
        
        return null;
    }

    /**
     * The email text body for invitation by email
     * @param defaultI18nParams The default i18n parameters with :
     * siteName the site name
     * email the mail
     * token the token
     * confirmUri the confirmation uri
     * siteTitle the site title
     * siteUrl the site url
     * @param language the language
     * @return the email text for invitation by email
     */
    public String getTextBodyForInvitationEmail(Map<String, I18nizableTextParameter> defaultI18nParams, String language)
    {
        if (StringUtils.isNotBlank(_textBodyKeyForInvitation))
        {
            return _i18nUtils.translate(_getI18nizableText(_textBodyKeyForInvitation, defaultI18nParams), language);
        }
        
        return null;
    }

    /**
     * The email html body for invitation by email
     * @param defaultI18nParams The default i18n parameters with :
     * siteName the site name
     * email the mail
     * token the token
     * confirmUri the confirmation uri
     * siteTitle the site title
     * siteUrl the site url
     * @param language the language
     * @return the email html for invitation by email
     */
    public String getHtmlBodyForInvitationEmail(Map<String, I18nizableTextParameter> defaultI18nParams, String language)
    {
        if (StringUtils.isNotBlank(_htmlBodyKeyForInvitation))
        {
            try
            {
                MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                        .withLanguage(language)
                        .withMessage(_getI18nizableText(_htmlBodyKeyForInvitation, defaultI18nParams));
                    
                Page signupPage = _getSignupPage(_projectManager.getCatalogSiteName(), language);
                if (signupPage != null)
                {
                    String signupUri = ResolveURIComponent.resolve("page", signupPage.getId(), false, true) + "?email=${email}&token=${token}";
                    bodyBuilder.withLink(signupUri, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_USER_INVITATION_MAIL_BODY_HTML_LINK_TITLE"));
                }
                
                return bodyBuilder.build();
            }
            catch (IOException e)
            {
                getLogger().warn("Faild to build HTML body for invitation email", e);
            }
        }
        
        return null;
    }
    
    /**
     * The email subject for invitation by email
     * @param defaultI18nParams The default i18n parameters
     * @param language the language
     * @return the email subject for invitation by email
     */
    public String getSubjectForInvitationAcceptedEmail (Map<String, I18nizableTextParameter> defaultI18nParams , String language)
    {
        if (StringUtils.isNotBlank(_subjectKeyForInvitationAccepted))
        {
            return _i18nUtils.translate(_getI18nizableText(_subjectKeyForInvitationAccepted, defaultI18nParams), language);
        }
        
        return null;
    }
    
    /**
     * The email text body for invitation by email
     * @param defaultI18nParams The default i18n parameters with :
     * @param language the language
     * @return the email text for invitation by email
     */
    public String getTextBodyForInvitationAcceptedEmail(Map<String, I18nizableTextParameter> defaultI18nParams, String language)
    {
        if (StringUtils.isNotBlank(_textBodyKeyForInvitationAccepted))
        {
            return _i18nUtils.translate(_getI18nizableText(_textBodyKeyForInvitationAccepted, defaultI18nParams), language);
        }
        
        return null;
    }
    
    /**
     * The email html body for invitation by email
     * @param project The project
     * @param defaultI18nParams The default i18n parameters with :
     * @param language the language
     * @return the email html for invitation by email
     */
    public String getHtmlBodyForInvitationAcceptedEmail(Project project, Map<String, I18nizableTextParameter> defaultI18nParams, String language)
    {
        if (StringUtils.isNotBlank(_htmlBodyKeyForInvitationAccepted))
        {
            try
            {
                MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                        .withLanguage(language)
                        .withMessage(_getI18nizableText(_htmlBodyKeyForInvitationAccepted, defaultI18nParams));
                    
                Set<Page> memberPages = _projectManager.getModulePages(project, MembersWorkspaceModule.MEMBERS_MODULE_ID);
                if (!memberPages.isEmpty())
                {
                    Page page = memberPages.iterator().next();
                    String memberPageUri = ResolveURIComponent.resolve("page", page.getId(), false, true);
                    bodyBuilder.withLink(memberPageUri, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_USER_INVITATION_ACCEPTED_MAIL_BODY_HTML_LINK_TITLE"));
                }
                
                return bodyBuilder.build();
            }
            catch (IOException e)
            {
                getLogger().warn("Faild to build HTML body for invitation accepted email", e);
            }
        }
        
        return null;
    }
    
    /**
     * Get the {@link I18nizableText} from the configured key and i18n parameters
     * @param fullI18nKey the configured i18n key
     * @param i18nParams the i18n parameters
     * @return the i18nizable text
     */
    protected I18nizableText _getI18nizableText(String fullI18nKey, Map<String, I18nizableTextParameter> i18nParams)
    {
        String catalogue = StringUtils.contains(fullI18nKey, ":") ? StringUtils.substringBefore(fullI18nKey, ":") : "plugin." + _pluginName;
        String i18nKey = StringUtils.contains(fullI18nKey, ":") ? StringUtils.substringAfter(fullI18nKey, ":") : fullI18nKey;
        
        return new I18nizableText(catalogue, i18nKey, i18nParams);
    }
    
}
