/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.favorites;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesExtensionPoint;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * About setting a project as a favorite project for a user
 */
public class FavoritesHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** the id of the user prefs for favorites preferences */
    public static final String USERPREFS_ID = "workspaces.favorites";
    
    /** The avalon role */
    public static final String ROLE = FavoritesHelper.class.getName();
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The user preferences manager */
    protected UserPreferencesManager _userPrefManager;
    /** The user preferences extension point */
    protected UserPreferencesExtensionPoint _userPrefEP;
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userPrefManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE + ".FO");
        _userPrefEP = (UserPreferencesExtensionPoint) manager.lookup(UserPreferencesExtensionPoint.ROLE + ".FO");
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Get the favorites projects of a user
     * @param userIdentity The user to consider
     * @return The non null list of favorite projects names
     */
    public Set<String> getFavorites(UserIdentity userIdentity)
    {
        return new HashSet<>(getPreferences(userIdentity));

    }
    
    /**
     * The current user change a favorite status for a given project
     * @param projectName The name of the project
     * @param favorite The new favorite status
     * @return true if successful
     */
    @Callable
    public boolean setFavoriteProject(String projectName, boolean favorite)
    {
        UserIdentity user = _currentUserProvider.getUser();
        try
        {
            List<String> preferences = new ArrayList<>(getPreferences(user));
            
            if (favorite && !preferences.contains(projectName))
            {
                preferences.add(projectName);
            }
            else if (!favorite && preferences.contains(projectName))
            {
                preferences.remove(projectName);
            }
    
            String preferencesAsString = _jsonUtils.convertObjectToJson(preferences);
                
            Map<String, String> values = _userPrefManager.getUnTypedUserPrefs(user, _getStorageContext(), Map.of());
            
            values.put(USERPREFS_ID, preferencesAsString);
            
            _userPrefManager.setUserPreferences(user, _getStorageContext(), Map.of(), values);
            
            return true;
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Could not change favorite status of user " + UserIdentity.userIdentityToString(user) + " on project " + projectName);
            return false;
        }
    }
    
    @SuppressWarnings("unchecked")
    private List<String> getPreferences(UserIdentity userIdentity)
    {
        try
        {
            String prefAsString = _userPrefManager.getUserPreferenceAsString(userIdentity, _getStorageContext(), Map.of(), USERPREFS_ID);
            if (prefAsString != null)
            {
                return (List) _jsonUtils.convertJsonToList(prefAsString);
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occurred while reading the preferences of user " + UserIdentity.userIdentityToString(userIdentity), e);
        }
        return List.of();
    }
    
    private String _getStorageContext()
    {
        return "/sites/" + _projectManager.getCatalogSiteName();
    }
}
