/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;


/**
 * Generator for the notification summary
 */
public class SummaryMailNotifierGenerator extends ActionResultGenerator implements Serviceable
{
    private ProjectManager _projectManager;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
    }
    
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        @SuppressWarnings("unchecked")
        Map<String, Object> parentContextAttr = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        String frequency = (String) parentContextAttr.get("frequency");
        
        @SuppressWarnings("unchecked")
        Map<String, Map<String, List<Map<String, Object>>>> activitiesByProject = (Map<String, Map<String, List<Map<String, Object>>>>) parentContextAttr.get("activities");
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("frequency", frequency);
        XMLUtils.startElement(contentHandler, "notification", attrs);
        
        for (Entry<String, Map<String, List<Map<String, Object>>>> entry : activitiesByProject.entrySet())
        {
            String projectName = entry.getKey();
            
            Project project = _projectManager.getProject(projectName);
            if (project != null)
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("name", projectName);
                attrs.addCDATAAttribute("title", project.getTitle());
                attrs.addCDATAAttribute("url", _projectManager.getProjectUrl(project, null));
                
                XMLUtils.startElement(contentHandler, "project", attrs);
                
                Map<String, List<Map<String, Object>>> activitiesByType = entry.getValue();
                
                for (Entry<String, List<Map<String, Object>>> entry2 : activitiesByType.entrySet())
                {
                    String eventType = entry2.getKey();
                    
                    attrs.clear();
                    attrs.addCDATAAttribute("type", eventType);
                    XMLUtils.startElement(contentHandler, "events", attrs);
                    
                    for (Map<String, Object> activityJSON : entry2.getValue())
                    {
                        _saxActivity(activityJSON);
                    }
                    
                    XMLUtils.endElement(contentHandler, "events");
                }
                
                XMLUtils.endElement(contentHandler, "project");
            }
        }
        
        XMLUtils.endElement(contentHandler, "notification");
        contentHandler.endDocument();
    }

    private void _saxActivity(Map<String, Object> activityJSON) throws ProcessingException, IOException, SAXException
    {
        XMLUtils.startElement(contentHandler, "event");
        
        for (Map.Entry<String, Object> entry : activityJSON.entrySet())
        {
            String key = entry.getKey();
            Object value = entry.getValue();

            saxMapItem(key, value);
        }
        
        XMLUtils.endElement(contentHandler, "event");
        
    }
    
    @Override
    protected void saxMapItem(String key, Object value) throws IOException, SAXException, ProcessingException
    {
        if (value instanceof Date)
        {
            Date date = (Date) value;
            XMLUtils.createElement(contentHandler, key, DateUtils.dateToString(date));
        }
        else
        {
            super.saxMapItem(key, value);
        }
    }
}
