/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.project.helper;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.tag.Tag;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.dom.AmetysNodeList;
import org.ametys.core.util.dom.MapElement;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.activities.activitystream.ActivityStreamClientInteraction;
import org.ametys.plugins.workspaces.alert.AlertWorkspaceModule;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.keywords.KeywordsDAO;
import org.ametys.plugins.workspaces.news.NewsWorkspaceModule;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.cocoon.I18nUtils;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.transformation.xslt.AmetysXSLTHelper;

/**
 * Helper component to be used from XSL stylesheets to get info related to projects.
 */
public class ProjectXsltHelper implements Serviceable, Contextualizable, LogEnabled
{
    private static Logger _logger;
    private static Context _context;
    private static SiteManager _siteManager;
    private static ProjectManager _projectManager;
    private static CategoryProviderExtensionPoint _categoryProviderEP;
    private static CategoryHelper _categoryHelper;
    private static I18nUtils _i18nUtils;
    private static ActivityStreamClientInteraction _activityStream;
    private static LanguagesManager _languagesManager;
    private static CurrentUserProvider _currentUserProvider;
    private static WorkspaceModuleExtensionPoint _workspaceModuleEP;
    private static RightManager _rightManager;
    private static KeywordsDAO _keywordsDAO;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _categoryHelper = (CategoryHelper) manager.lookup(CategoryHelper.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(org.ametys.core.util.I18nUtils.ROLE);
        _activityStream = (ActivityStreamClientInteraction) manager.lookup(ActivityStreamClientInteraction.ROLE);
        _languagesManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _keywordsDAO = (KeywordsDAO) manager.lookup(KeywordsDAO.ROLE);
    }
    
    /**
     * Returns the current project
     * @return the current project
     */
    public static String project()
    {
        Project project = null;
        
        Request request = ContextHelper.getRequest(_context);
        String siteName = (String) request.getAttribute("site");
        
        if (StringUtils.isNotEmpty(siteName) && _siteManager.hasSite(siteName))
        {
            Site site = _siteManager.getSite(siteName);
            List<Project> projects = _projectManager.getProjectsForSite(site);
            
            project = !projects.isEmpty() ? projects.get(0) : null;
        }
        
        if (project == null && _logger.isWarnEnabled())
        {
            String warnMsg = String.format("No project was found for site '%s'.", siteName);
            _logger.warn(warnMsg);
        }
        
        return project != null ? project.getName() : null;
    }
    
    /**
     * Returns the information of current project
     * @return the information of current project
     */
    public static MapElement projectInfo()
    {
        return projectInfo(project());
    }
    
    /**
     * Returns the project information
     * @param projectName The project name
     * @return the project information
     */
    public static MapElement projectInfo(String projectName)
    {
        Map<String, Object> projectInfo = new HashMap<>();
        
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            projectInfo.put("name", project.getName());
            projectInfo.put("id", project.getId());
            projectInfo.put("title", project.getTitle());
            projectInfo.put("description", project.getDescription());
            projectInfo.put("creationDate", _projectCreationDate(project));
            projectInfo.put("lastActivityDate", projectLastActivityDate(project.getName()));
            projectInfo.put("categoryLabel", _projectCategoryLabel(project));
            projectInfo.put("categoryColor", _projectCategoryColor(project));
            projectInfo.put("keyword", _projectKeywords(project));
        }
        
        return new MapElement("project", projectInfo);
    }
    
    /**
     * Returns the title of the current project
     * @return the title of the current project
     */
    public static String projectTitle()
    {
        return projectTitle(project());
    }
    
    /**
     * Returns the title of the given project
     * @param projectName The project to consider
     * @return the title of the given project or empty otherwise
     */
    public static String projectTitle(String projectName)
    {
        String title = "";
        
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            title = project.getTitle();
        }
        
        return title;
    }
    
    /**
     * Returns the description of the current project
     * @return the description of the current project
     */
    public static String projectDescription()
    {
        return projectDescription(project());
    }
    
    /**
     * Returns the description of the given project
     * @param projectName The project to consider
     * @return the description of the given project or empty otherwise
     */
    public static String projectDescription(String projectName)
    {
        String title = "";
        
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            title = project.getDescription();
        }
        
        return title;
    }
    
    /**
     * Returns the keywords of the current project
     * @return keywords of the current project
     */
    public static NodeList projectKeywords()
    {
        return projectKeywords(project());
    }
    
    /**
     * Returns the project keywords
     * @param projectName the project name
     * @return the project keywords
     */
    public static NodeList projectKeywords(String projectName)
    {
        List<MapElement> keywordsElmts = new ArrayList<>();
        
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            List<Map<String, String>> keywords = _projectKeywords(project);
            for (Map<String, String> keyword : keywords)
            {
                keywordsElmts.add(new MapElement("keyword", keyword));
            }
        }
        
        return new AmetysNodeList(keywordsElmts);
    }
    
    private static List<Map<String, String>> _projectKeywords(Project project)
    {
        List<Map<String, String>> keywords = new ArrayList<>();
        
        for (String keyword : project.getKeywords())
        {
            Tag tag = _keywordsDAO.getTag(keyword, Collections.emptyMap());
            if (tag != null)
            {
                String title = _i18nUtils.translate(tag.getTitle(), AmetysXSLTHelper.lang());
                keywords.add(Map.of("title", title, "name", keyword));
            }
        }
        
        return keywords;
    }
    
    /**
     * Returns the creation date of the current project
     * @return the creation date of the current project at the ISO format or empty otherwise
     */
    public static String projectCreationDate()
    {
        return projectCreationDate(project());
    }
    
    /**
     * Returns the creation date of the given project
     * @param projectName The project to consider
     * @return the creation date of the given project at the ISO format or empty otherwise
     */
    public static String projectCreationDate(String projectName)
    {
        String creationDate = "";
        
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            creationDate = _projectCreationDate(project);
        }
        
        return creationDate;
    }
    
    private static String _projectCreationDate(Project project)
    {
        ZonedDateTime date = project.getCreationDate();
        return date.format(DateUtils.getISODateTimeFormatter());
    }
    
    /**
     * Return the color associated to the first category associated to the current project
     * @return the hexa code color or the default color if no color is associated to the first category or if not category is associated.
     * &lt;color&gt;
     *     &lt;main&gt;#FFFFFF&lt;/main&gt;
     *     &lt;text&gt;#000000&lt;/text&gt;
     * &lt;/color&gt;
     */
    public static MapElement projectCategoryColor()
    {
        return projectCategoryColor(project());
    }
    
    /**
     * Return the color associated to the first category associated to the given project
     * @param projectName The project to consider
     * @return the hexa code color or the default color if no color is associated to the first category or if not category is associated
     * &lt;color&gt;
     *     &lt;main&gt;#FFFFFF&lt;/main&gt;
     *     &lt;text&gt;#000000&lt;/text&gt;
     * &lt;/color&gt;
     */
    public static MapElement projectCategoryColor(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        Map<String, String> color = _projectCategoryColor(project);
        return new MapElement("color", color);
    }
    
    private static Map<String, String> _projectCategoryColor(Project project)
    {
        Category category = _getFirstCategory(project);
        return _categoryColor(category);
    }
    
    /**
     * Return the color associated to the category
     * @param categoryId category id
     * @return the hexa code color or the default color if no color is associated to the category or if null
     * &lt;color&gt;
     *     &lt;main&gt;#FFFFFF&lt;/main&gt;
     *     &lt;text&gt;#000000&lt;/text&gt;
     * &lt;/color&gt;
     */
    public static MapElement categoryColor(String categoryId)
    {
        Category category = _categoryProviderEP.getTag(categoryId, null);
        return categoryColor(category);
    }
    
    /**
     * Return the color associated to the category
     * @param category category
     * @return the hexa code color or the default color if no color is associated to the category or if null
     * &lt;color&gt;
     *     &lt;main&gt;#FFFFFF&lt;/main&gt;
     *     &lt;text&gt;#000000&lt;/text&gt;
     * &lt;/color&gt;
     */
    public static MapElement categoryColor(Category category)
    {
        Map<String, String> color = _categoryColor(category);
        return new MapElement("color", color);
    }
    
    private static Map<String, String> _categoryColor(Category category)
    {
        return _categoryHelper.getCategoryColor(category);
    }
    
    private static Category _getFirstCategory(Project project)
    {
        if (project != null)
        {
            for (String categoryId : project.getCategories())
            {
                Category category = _categoryProviderEP.getTag(categoryId, null);
                if (category != null)
                {
                    return category;
                }
            }
        }
        return null;
    }
    
    
    /**
     * Return the color associated to the first category associated to the current project
     * @return the hexa code color or the default color if no color is associated to the first category or if not category is associated
     */
    public static String projectCategoryLabel()
    {
        return projectCategoryLabel(project());
    }
    
    /**
     * Return the color associated to the first category associated to the given project
     * @param projectName The project to consider
     * @return the label or empty if no category
     */
    public static String projectCategoryLabel(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            return _projectCategoryLabel(project);
        }
        return null;
    }
    
    private static String _projectCategoryLabel(Project project)
    {
        Category category = _getFirstCategory(project);
        if (category != null)
        {
            return _i18nUtils.translate(category.getTitle(), AmetysXSLTHelper.lang());
        }
        return null;
    }
    
    /**
     * Get the date of last activity for current project
     * @return the formatted date of last activity or creation date if there is no activity yet
     */
    public static String projectLastActivityDate()
    {
        return projectLastActivityDate(project());
    }
    
    /**
     * Get the date of last activity for a given project
     * @param projectName the project's name
     * @return the formatted date of last activity or creation date if there is no activity yet
     */
    public static String projectLastActivityDate(String projectName)
    {
        ZonedDateTime lastDate = _activityStream.getDateOfLastActivity(projectName, Arrays.asList(ObservationConstants.EVENT_MEMBER_ADDED, ObservationConstants.EVENT_MEMBER_DELETED));
        return lastDate != null ? lastDate.format(DateUtils.getISODateTimeFormatter()) : projectCreationDate();
    }
    
    /**
     * True if the resource comes from workspaces
     * @param resourceId the resource id
     * @return true if the resource comes from workspaces
     */
    public static boolean isResourceFromWorkspace(String resourceId)
    {
        return _projectManager.getParentProject(resourceId) != null;
    }
    
    /**
     * Get the site of a project's resource
     * @param projectResourceId The resource id
     * @return The site &lt;site id="site://xxx" name="siteName"&gt;&lt;title&gt;Site's titleX&lt;/title&gt;&lt;url&gt;http://...&lt;/url&gt;/site&gt;
     */
    public static Node resourceSite(String projectResourceId)
    {
        Project project = _projectManager.getParentProject(projectResourceId);
        if (project != null)
        {
            Site site = project.getSite();
            
            Map<String, String> attributes = new HashMap<>();
            attributes.put("name", site.getName());
            attributes.put("id", site.getId());
            
            Map<String, String> values = new HashMap<>();
            values.put("title", site.getTitle());
            values.put("url", site.getUrl());
            
            return new MapElement("site", attributes, values);
        }
        
        return null;
    }
    
    /**
     * Get the id of news root's page
     * @return the id of news root's page or null if not exists
     */
    public static String getNewsRootPageId()
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace (news root's page can not be publish yet as it is a node page)
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            return getModulePage(NewsWorkspaceModule.NEWS_MODULE_ID);
        }
        finally
        {
            // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Get the id of alerts root's page
     * @return the id of alerts root's page or null if not exists
     */
    public static String getAlertsRootPageId()
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace (news root's page can not be publish yet as it is a node page)
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            return getModulePage(AlertWorkspaceModule.ALERT_MODULE_ID);
        }
        finally
        {
            // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Get the module page.
     * Be careful, the read access is not check !
     * @param moduleId the module id
     * @return the module page
     */
    public static String getModulePage(String moduleId)
    {
        String projectName = project();
        Project project = _projectManager.getProject(projectName);
        
        Set<Page> newsRootPages = _projectManager.getModulePages(project, moduleId);
        Iterator<Page> it = newsRootPages.iterator();
        if (it.hasNext())
        {
            return it.next().getId();
        }
        
        return null;
    }
    
    /**
     * Get the translated label associated to the code
     * @param code The language code
     * @param language The language in which translate the label
     * @return The label associated to the code
     */
    public static String getLanguageLabel(String code, String language)
    {
        I18nizableText label = _languagesManager.getLanguage(code).getLabel();
        return _i18nUtils.translate(label, language);
    }
    
    /**
     * Get the language that will be the default value of the available values.
     * First choice will be the page language, 'en' otherwise, first available language otherwise.
     * @param language The current page language
     * @return A language code
     */
    public static String computeDefaultLanguage(String language)
    {
        Request request = ContextHelper.getRequest(_context);
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        ModelAwareDataHolder dataHolder = zoneItem.getServiceParameters();
        
        String[] availableLanguages = dataHolder.getValue("availableLanguages");
        if (Arrays.stream(availableLanguages).anyMatch(l -> l.equals(language)))
        {
            return language;
        }
        else if (Arrays.stream(availableLanguages).anyMatch(l -> l.equals("en")))
        {
            return "en";
        }
        else
        {
            return availableLanguages[0];
        }
    }
    
    /**
     * Determines if the current user is a manager of current project
     * @return true if the current user is manager
     */
    public static boolean isManager()
    {
        UserIdentity user = _currentUserProvider.getUser();
        String projectName = project();
        return _projectManager.isManager(projectName, user);
    }
    
    /**
     * Determines if a user is a manager
     * @param login the user's login
     * @param populationId the user's population
     * @return true if the user is manager
     */
    public static boolean isManager(String login, String populationId)
    {
        String projectName = project();
        UserIdentity user = new UserIdentity(login, populationId);
        return _projectManager.isManager(projectName, user);
    }
    
    /**
     * Determines if the current user is a manager for the project
     * @param projectName the project's name
     * @return true if the current user is manager
     */
    public static boolean isManager(String projectName)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return _projectManager.isManager(projectName, user);
    }
    
    /**
     * Determines if the given user is a manager for the project
     * @param projectName the project's name
     * @param login the user's login
     * @param populationId the user's population
     * @return true if the user is manager
     */
    public static boolean isManager(String projectName, String login, String populationId)
    {
        UserIdentity user = new UserIdentity(login, populationId);
        return _projectManager.isManager(projectName, user);
    }
    
    
    /**
     * Returns true if the current user has the specified right on the specified workspace module
     * @param rightId Right Id
     * @param moduleId Module Id
     * @return true if the current user has the specified right on the specified module
     */
    public static boolean hasRightOnModule(String rightId, String moduleId)
    {
        String projectName = project();
        WorkspaceModule module = _workspaceModuleEP.getModule(moduleId);
        if (module != null && projectName != null)
        {
            Project project = _projectManager.getProject(projectName);
            if (project != null)
            {
                ModifiableResourceCollection moduleRoot = module.getModuleRoot(project, false);
                if (moduleRoot != null)
                {
                    return  _rightManager.currentUserHasRight(rightId, moduleRoot) == RightResult.RIGHT_ALLOW;
                }
            }
        }
        
        return false;
    }

    /**
     * Returns true if current user can access to the back-office
     * @return true if current user can access to the back-office
     */
    public static boolean canAccessBO()
    {
        String projectName = project();
        Project project = _projectManager.getProject(projectName);
        return project != null ? _projectManager.canAccessBO(project) : false;
    }
    
    /**
     * Returns true if current user can leave the project
     * @return true if current user can leave the project
     */
    public static boolean canLeaveProject()
    {
        String projectName = project();
        Project project = _projectManager.getProject(projectName);
        return project != null ? _projectManager.canLeaveProject(project) : false;
    }
}
