/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search;

import java.io.IOException;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Generator for modular search service
 *
 */
public class ModularSearchGenerator extends ServiceableGenerator
{
    /** Search Module Extension Point */
    protected SearchModuleExtensionPoint _searchModuleEP;
    
    /** Source Resolver */
    protected SourceResolver _sourceResolver;

    /** The project manager */
    protected ProjectManager _projectManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;
    
    /** The category provider */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    
    /** The category helper */
    protected CategoryHelper _categoryHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _searchModuleEP = (SearchModuleExtensionPoint) smanager.lookup(SearchModuleExtensionPoint.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _projectMemberManager = (ProjectMemberManager) smanager.lookup(ProjectMemberManager.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _categoryHelper = (CategoryHelper) smanager.lookup(CategoryHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        boolean withResults = parameters.getParameterAsBoolean("withResults", false);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String moduleId = request.getParameter("moduleId");
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "search");
        
        saxFilters(request);
        
        if (moduleId != null)
        {
            // Load more results
            SearchModule searchModule = _searchModuleEP.getExtension(moduleId);
            int offset = parameters.getParameterAsInteger("offset", 0);
            saxSearchModule(searchModule, withResults, offset);
        }
        else 
        {
            List<Project> availableProjects = getAvailableProjects();
            saxProjects(availableProjects);
            saxCategories(availableProjects);
            saxSearchModules(withResults);
        }
        
        XMLUtils.endElement(contentHandler, "search");
        contentHandler.endDocument();
    }
    
    

    /**
     * SAX the existing search module
     * @param withResults true to set results
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxSearchModules(boolean withResults) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "modules");
        
        Map<Integer, SearchModule> searchModules = new TreeMap<>();
        for (String extensionId : _searchModuleEP.getExtensionsIds())
        {
            SearchModule searchModule = _searchModuleEP.getExtension(extensionId);
            searchModules.put(searchModule.getOrder(), searchModule);
        }
        
        for (SearchModule searchModule : searchModules.values())
        {
            saxSearchModule(searchModule, withResults, 0);
        }
        
        XMLUtils.endElement(contentHandler, "modules");
    }
    
    /**
     * SAX the existing search module
     * @param searchModule The search module
     * @param withResults true to set results
     * @param offset the offset search
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxSearchModule(SearchModule searchModule, boolean withResults, int offset) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", searchModule.getId());
        attrs.addCDATAAttribute("url", searchModule.getSearchUrl());
        attrs.addCDATAAttribute("order", String.valueOf(searchModule.getOrder()));
        attrs.addCDATAAttribute("offset", String.valueOf(offset));
        attrs.addCDATAAttribute("limit", String.valueOf(searchModule.getLimit()));
        attrs.addCDATAAttribute("minLimit", String.valueOf(searchModule.getMinLimit()));
        
        XMLUtils.startElement(contentHandler, "module", attrs);
        searchModule.getTitle().toSAX(contentHandler, "title");
        
        if (withResults)
        {
            saxSearchModuleResults(searchModule, offset);
        }
        XMLUtils.endElement(contentHandler, "module");
    }
    /**
     * Sax the results of a search module
     * @param searchModule the search module
     * @param offset the offset search
     */
    protected void saxSearchModuleResults(SearchModule searchModule, int offset)
    {
        SitemapSource src = null; 
        try
        {
            String uri = "cocoon://" + searchModule.getSearchUrl();
            
            Map<String, Object> params = new HashMap<>();
            params.put("offset", offset);
            params.put("limit", searchModule.getLimit());
            params.put("minLimit", searchModule.getMinLimit());
            
            src = (SitemapSource) _sourceResolver.resolveURI(uri, null, params);
            src.toSAX(new IgnoreRootHandler(contentHandler));
        }
        catch (SAXException | IOException e)
        {
            getLogger().error("The search failed for module '" + searchModule.getId() + "'", e);
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }
    
    /**
     * Get the available projects (the user's project)
     * @return the available projects for search
     */
    protected List<Project> getAvailableProjects()
    {
        UserIdentity user = _currentUserProvider.getUser();

        Function<Project, String> getProjectTitle = Project::getTitle;
        Comparator<Project> projectTitleComparator = Comparator.comparing(getProjectTitle.andThen(StringUtils::stripAccents), String.CASE_INSENSITIVE_ORDER);
        
        return  _projectManager.getUserProjects(user)
                .keySet()
                .stream()
                .sorted(projectTitleComparator)
            .collect(Collectors.toList());
    }
    
    /**
     * SAX the active filters
     * @param request the request
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxFilters(Request request) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "filters");
        
        String textfield = request.getParameter("textfield");
        if (StringUtils.isNotBlank(textfield))
        {
            XMLUtils.createElement(contentHandler, "textfield", textfield);
        }
        
        String[] categories = request.getParameterValues("category");
        if (categories != null)
        {
            for (String category : categories)
            {
                XMLUtils.createElement(contentHandler, "category", category);
            }
        }
        
        
        String[] projects = request.getParameterValues("project");
        if (projects != null)
        {
            for (String project : projects)
            {
                XMLUtils.createElement(contentHandler, "project", project);
            }
        }
        
        XMLUtils.endElement(contentHandler, "filters");
    }
    
    
    /**
     * SAX the available projects
     * @param projects the projects to sax
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxProjects(List<Project> projects) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "projects");
        for (Project project : projects)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", project.getId());
            attrs.addCDATAAttribute("name", project.getName());
            XMLUtils.createElement(contentHandler, "project", attrs, project.getTitle());
        }
        XMLUtils.endElement(contentHandler, "projects");
    }
    
    /**
     * SAX the categories
     * @param projects the available projects
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxCategories(List<Project> projects) throws SAXException
    {
        // Get the root categories from the available projects
        Set<Category> rootCategories = projects.stream()
            .map(Project::getCategories)
            .flatMap(Collection::stream)
            .map(id -> _categoryProviderEP.getTag(id, null))
            .filter(Objects::nonNull)
            .map(c -> _getRootCategory(c))
            .collect(Collectors.toSet());
        
        XMLUtils.startElement(contentHandler, "categories");
        for (Category category : rootCategories)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", category.getId());
            attrs.addCDATAAttribute("name", category.getName());
            XMLUtils.startElement(contentHandler, "category", attrs);
            
            Map<String, String> colors = _categoryHelper.getCategoryColor(category);
            XMLUtils.createElement(contentHandler, "color", colors.get("main"));
            category.getTitle().toSAX(contentHandler, "title");
            XMLUtils.endElement(contentHandler, "category");
        }
        XMLUtils.endElement(contentHandler, "categories");
    }
    
    private Category _getRootCategory(Category category)
    {
        Category parent = category;
        while (parent.getParent() != null)
        {
            parent = parent.getParent();
        }
        return parent;
    }
    
}
