/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.jcr;

import javax.jcr.Node;

import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.DefaultAmetysObjectFactory;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.workspaces.data.type.ModelItemTypeExtensionPoint;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link AmetysObjectFactory} for handling {@link JCRTask}.
 */
public class JCRTaskFactory extends DefaultAmetysObjectFactory
{
    /** JCR nodetype for tasks */
    public static final String TASK_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":task";
    
    /** The task model */
    private Model _taskModel;
    
    @Override
    public JCRTask getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new JCRTask(node, parentPath, this);
    }
    
    /**
     * Get the task model
     * @return the task model
     */
    public Model getTaskModel()
    {
        if (_taskModel == null)
        {
            try
            {
                String role = ModelItemTypeExtensionPoint.ROLE_TASK;
                _taskModel = Model.of(
                        "task.model.id", 
                        "task.model.family.id",
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_TASKSLISTID, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_TASKSLISTPOSITION, false, ModelItemTypeConstants.LONG_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_LABEL, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_DESCRIPTION, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_STARTDATE, false, ModelItemTypeConstants.DATE_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_DUEDATE, false, ModelItemTypeConstants.DATE_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_ISCLOSED, false, ModelItemTypeConstants.BOOLEAN_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_CLOSEAUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_CLOSEDATE, false, ModelItemTypeConstants.DATE_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_AUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_ASSIGNMENTS, true, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_CREATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_LASTMODIFIED, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRTask.ATTRIBUTE_ATTACHMENTS, true, org.ametys.cms.data.type.ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID, role),
                        RepeaterDefinition.of(JCRTask.ATTRIBUTE_CHECKLIST, role,
                                DefaultElementDefinition.of(JCRTask.ATTRIBUTE_CHECKLIST_LABEL, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                                DefaultElementDefinition.of(JCRTask.ATTRIBUTE_CHECKLIST_ISCHECKED, false, ModelItemTypeConstants.BOOLEAN_TYPE_ID, role))
                        );
            }
            catch (Exception e) 
            {
                getLogger().error("An error occurred getting the task model", e);
                throw new RuntimeException("An error occurred getting the task model", e);
            }
        }
        return _taskModel;
    }
}
