/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.wall;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.content.WebContentDAO;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneDAO;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.search.query.SiteQuery;

import com.google.common.collect.ImmutableSet;

/**
 * Workspace module for wall content
 */
public class WallContentModule extends AbstractWorkspaceModule
{
    /** The id of calendar module */
    public static final String WALLCONTENT_MODULE_ID = WallContentModule.class.getName();
    
    /** The id of wall content service */
    public static final String WALLCONTENT_SERVICE_ID = "org.ametys.web.service.SearchService";
    
    /** Search service content types */
    protected static final String[] SEARCH_SERVICE_CONTENT_TYPES = new String[] {WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID};
    
    /** Search service returnables */
    protected static final String[] SEARCH_SERVICE_RETURNABLES = new String[] {"org.ametys.web.frontoffice.search.metamodel.impl.ContentReturnable"};
    
    /** Search service sorts */
    protected static final String[] SEARCH_SERVICE_SORTS = new String[] {"{\"name\":\"ContentReturnable$ContentSearchable$indexingField$org.ametys.plugins.workspaces.Content.wallContent$pinned\",\"sort\":\"DESC\"}", "{\"name\":\"ContentReturnable$ContentSearchable$systemProperty$creationDate\",\"sort\":\"DESC\"}"};
    
    /** Search service contexts */
    protected static final String[] SEARCH_SERVICE_CONTEXTS = new String[] {"{\"sites\":\"{\\\"context\\\":\\\"CURRENT_SITE\\\",\\\"sites\\\":[]}\",\"search-sitemap-context\":\"{\\\"context\\\":\\\"CURRENT_SITE\\\",\\\"page\\\":null}\",\"context-lang\":\"CURRENT\",\"tags\":[]}"};
    
    /** Search service xslt */
    protected static final String SEARCH_SERVICE_XSLT = "pages/services/search/wall-content.xsl";
    
    /** Workspaces wallcontent node name */
    private static final String __WORKSPACES_WALLCONTENT_NODE_NAME = "wallcontent";

    private static final String __WALLCONTENT_NUMBER_HEADER_ID = __WORKSPACES_WALLCONTENT_NODE_NAME + "$wall_content_number";
    
    /** The zone DAO */
    protected ZoneDAO _zoneDAO;
    /** The content DAO */
    protected WebContentDAO _contentDAO;
    /** The content searcher factory */
    protected ContentSearcherFactory _contentSearcherFactory;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _zoneDAO = (ZoneDAO) manager.lookup(ZoneDAO.ROLE);
        _contentDAO = (WebContentDAO) manager.lookup(WebContentDAO.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
    }
    
    public String getId()
    {
        return WALLCONTENT_MODULE_ID;
    }

    public String getModuleName()
    {
        return __WORKSPACES_WALLCONTENT_NODE_NAME;
    }
    
    public int getOrder()
    {
        return ORDER_WALLCONTENT;
    }

    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of(ObservationConstants.EVENT_WALLCONTENT_ADDED, org.ametys.cms.ObservationConstants.EVENT_CONTENT_COMMENT_VALIDATED);
    }

    @Override
    protected String getModulePageName()
    {
        return "index";
    }

    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_WALLCONTENT_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_WALLCONTENT_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        // The module page is the index page, which already exists
        return null;
    }

    @Override
    protected void initializeModulePage(ModifiablePage modulePage)
    {
        ModifiableZone defaultZone;
        if (modulePage.hasZone("default"))
        {
            defaultZone = modulePage.getZone("default");
        }
        else
        {
            defaultZone = modulePage.createZone("default");            
        }
        
        boolean hasService = defaultZone.getZoneItems().stream().anyMatch(zi -> WALLCONTENT_SERVICE_ID.equals(zi.getServiceId()));
        
        if (!hasService)
        {
            ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
            defaultZoneItem.setType(ZoneType.SERVICE);
            defaultZoneItem.setServiceId(WALLCONTENT_SERVICE_ID);
            
            ModifiableModelAwareDataHolder serviceDataHolder = defaultZoneItem.getServiceParameters();
            serviceDataHolder.setValue("header", null);
            serviceDataHolder.setValue("contentTypes", SEARCH_SERVICE_CONTENT_TYPES);
            serviceDataHolder.setValue("returnables", SEARCH_SERVICE_RETURNABLES);
            serviceDataHolder.setValue("initialSorts", SEARCH_SERVICE_SORTS);
            serviceDataHolder.setValue("contexts", SEARCH_SERVICE_CONTEXTS);
            serviceDataHolder.setValue("resultsPerPage", 15);
            serviceDataHolder.setValue("rightCheckingMode", "none");
            serviceDataHolder.setValue("resultPlace", "ABOVE_CRITERIA");
            serviceDataHolder.setValue("launchSearchAtStartup", true);
            serviceDataHolder.setValue("rss", false);
            serviceDataHolder.setValue("contentView", "main");
            serviceDataHolder.setValue("xslt", SEARCH_SERVICE_XSLT);
        }
    }

    /*
     * Override because the page should already exist, the parent returns null in this case
     */
    @Override
    protected ModifiablePage _createModulePage(Project project, Sitemap sitemap, String name, I18nizableText pageTitle, String skinTemplate)
    {
        if (sitemap.hasChild(name))
        {
            return sitemap.getChild(name);
        }
        else
        {
            return super._createModulePage(project, sitemap, name, pageTitle, skinTemplate);
        }
    }
    
    @Override
    protected void _deletePages(Project project)
    {
        // Nothing. Index page should not be deleted.
    }
    
    @Override
    protected void _internalDeactivateModule(Project project)
    {
        // Remove wall service
        _removeWallService(project);
    }
    
    @Override
    protected void _internalDeleteData(Project project)
    {
        // Remove wall service
        _removeWallService(project);
        
        // Delete wall contents
        _deleteWallContents(project);
    }
    
    private void _deleteWallContents(Project project)
    {
        Expression cTypeExpr = new ContentTypeExpression(Operator.EQ, WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID);
        
        Expression siteExpr = new StringExpression("site", Operator.EQ, project.getSite().getName());
        
        Expression expr = new AndExpression(cTypeExpr, siteExpr);
        
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(expr);
        
        List<String> contentIds = _resolver.query(xPathQuery)
            .stream()
            .map(AmetysObject::getId)
            .collect(Collectors.toList());
        
        _contentDAO.deleteContents(contentIds, true);
    }
    
    private void _removeWallService(Project project)
    {
        List<Page> modulePages = _getModulePages(project);
        for (Page page : modulePages)
        {
            if (page.hasZone("default"))
            {
                _projectManager.untagProjectPage((ModifiablePage) page, getModuleRoot(project, false));
                
                ModifiableZone defaultZone = ((ModifiablePage) page).getZone("default");
                
                Set<String> zoneItemIds = defaultZone.getZoneItems()
                        .stream()
                        .filter(zi -> WALLCONTENT_SERVICE_ID.equals(zi.getServiceId()))
                        .map(ZoneItem::getId)
                        .collect(Collectors.toSet());

                for (String zoneItemId : zoneItemIds)
                {
                    _zoneDAO.removeZoneItem(zoneItemId);
                }
            }
        }
    }
    
    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    { 
        if (isActive)
        {
            Map<String, Object> statistics = new HashMap<>();
            try
            {
                AmetysObjectIterable<Content> results = _contentSearcherFactory.create(WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID)
                        .search(new SiteQuery(project.getName()));
                statistics.put(__WALLCONTENT_NUMBER_HEADER_ID, results.getSize());
            }
            catch (Exception e)
            {
                getLogger().error("Error searching wall content in project " + project.getId(), e);
            }
            return statistics;
        }
        else
        {
            return Map.of(__WALLCONTENT_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }

    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__WALLCONTENT_NUMBER_HEADER_ID, new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_WALL_CONTENT_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }

    @Override
    protected long _getModuleSize(Project project)
    {
        try
        {
            AmetysObjectIterable<Content> resultsIterable = _contentSearcherFactory.create(WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID)
                    .search(new SiteQuery(project.getName()));

            return resultsIterable.stream()
                .map(content -> content.getValue("illustration/image"))
                .filter(Objects::nonNull)
                .filter(Binary.class::isInstance)
                .map(Binary.class::cast)
                .mapToLong(Binary::getLength)
                .sum();
        }
        catch (Exception e)
        {
            getLogger().error("Error searching wall content images in project " + project.getId(), e);
            return -1;
        }
    }

    @Override
    protected boolean _showModuleSize()
    {
        return true;
    }
}
