/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import Vue from 'vue';

export const mutations = {
  SET_USER_RIGHTS(state, data) {
    state.userRights = data;
  },
  SET_CATEGORY_THREAD_COUNT(state, data) {
    if (state.categoryThreadsCount == null) {
      state.categoryThreadsCount = { [data.id]: data.count };
    } else {
      Vue.set(state.categoryThreadsCount, data.id, data.count);
    }
  },
  SET_THREAD_IDS(state, threadIds) {
    state.threadIds = threadIds;
  },
  SET_THREADS(state, data) {
    state.threadsPagination = data.pagination;
    if (data.concatenate)
    {
      state.threads = [...state.threads, ...data.threads];
    }
    else
    {
      state.threads = data.threads;
    }
  },
  ADD_CATEGORY_THREAD_COUNT(state, category) {
    if (category) {
        state.categoryThreadsCount[category] = (state.categoryThreadsCount[category] || 0) + 1;
    }
  },
  ADD_THREAD(state, thread) {

    if (thread.passFilter)
    {
        // Add the new thread on the top
        state.threads.unshift(thread);
        state.threadIds.unshift(thread.id);
        // If there are still threads in the server, remove the last to keep same amount of threads
        if (state.threadsPagination.page != state.threadsPagination.lastPage)
        {
          // If there are still threads in the server, remove the last to keep same amount of threads
          if (state.threadsPagination.page != state.threadsPagination.lastPage)
          {
            state.threads.pop();
          }
        }
    }
  },
  SET_THREAD(state, data) {
    state.thread = data || {};
  },
  RESET_THREAD(state, data) {
    state.thread = {
      category: data.category,
      tags: [],
      attachments: [],
    };
  },
  UPDATE_THREAD(state, thread) {
    state.thread = thread;
    // update category counts if catageory changes
    const previous = state.initialThreadCategory;
    const newCategory = thread.category;
    const counts = state.categoryThreadsCount;
    if (previous != newCategory && counts && counts[previous] != undefined) {
      state.categoryThreadsCount[previous] = state.categoryThreadsCount[previous] > 0 ? state.categoryThreadsCount[previous] - 1 : 0;
      state.categoryThreadsCount[newCategory] = (state.categoryThreadsCount[newCategory] || 0) + 1;
    }

    // update tags state if new/updated thread has new tags
    let newTags = [];
    thread.tags.forEach((tag) => {
      const found = state.tags.find((t) => t.id == tag.id);
      if (!found) newTags.push(tag);
    });

    if (newTags.length > 0) state.tags = [...state.tags, ...newTags];

    // update threads state
    // If thread still pass filters, update the threads to take changes into account
    if (thread.passFilter)
    {
      state.threads = state.threads.map(u => u.id !== thread.id ? u : thread);
    }
    // If thread does not pass filter anymore, remove it from list of threads
    else
    {
      state.threads = state.threads.filter(s=> s.id != thread.id);
    }
  },
  SET_INITIAL_THREAD_CATEGORY(state, data) {
    state.initialThreadCategory = data;
  },
  REMOVE_THREAD(state, thread) {
    state.threads = state.threads.filter(s => s.id != thread.id);
    state.threadIds = state.threadIds.filter(threadId => threadId != thread.id);
    state.categoryThreadsCount[thread.category] = state.categoryThreadsCount[thread.category] > 0 ? state.categoryThreadsCount[thread.category] - 1 : 0;
  },
  SET_USED_TAGS(state, usedTags) {
    state.usedTags = usedTags;
    if (state.filters.tag && !usedTags.map(tag => tag.name).includes(state.filters.tag))
    {
      delete state.filters.tag;
    }
  },
  SET_FILTERS(state, val) {
    state.filters = val;
  },
  UPDATE_THREAD_WITH_NOTIFICATION(state, data) {

    (state.threads || []).forEach((thread) => {
      if (thread.id == data.id)
      {
        thread.hasUnopenedThreadNotification = false;
        thread.unreadComments = 0;
      }
    });

    if (state.thread && state.thread.id == data.id) {
      state.thread.hasUnopenedThreadNotification = false;
      state.thread.unreadComments = 0;
    }

    // if we filter by notifications, remove thread from threads list as the thread does not have a notification anymore
    if (data.filterByNotification)
    {
      state.threads = state.threads.filter(s=> s.id != data.id);
    }
  },

  MAKE_COMMENTS_READ(state, data) {

    (state.threads || []).forEach((thread) => {
      if (thread.id == data.id)
      {
        thread.unreadComments = thread.unreadComments - data.commentIds.length;
      }
    });

    if (state.thread && state.thread.id == data.id) {
      state.thread.unreadComments = state.thread.unreadComments - data.commentIds.length;
      state.thread.comments.forEach(c=>{
        if(data.commentIds.includes(c.id)) c.unread = false;
      })
    }

  },
};
