/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.collections.ListUtils;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollectionFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.calendars.Calendar.CalendarVisibility;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventJSONHelper;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEvent;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEventFactory;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

import com.google.common.collect.ImmutableSet;

/**
 * Helper component for managing calendars
 */
public class CalendarWorkspaceModule extends AbstractWorkspaceModule implements Configurable
{
    /** The id of calendar module */
    public static final String CALENDAR_MODULE_ID = CalendarWorkspaceModule.class.getName();
    
    /** Workspaces calendars node name */
    private static final String __WORKSPACES_CALENDARS_NODE_NAME = "calendars";

    /** Workspaces root tasks node name */
    private static final String __WORKSPACES_CALENDARS_ROOT_NODE_NAME = "calendars-root";
    
    /** Workspaces root tasks node name */
    private static final String __WORKSPACES_CALENDAR_RESOURCES_ROOT_NODE_NAME = "calendar-resources-root";
    
    /** Workspaces root tasks node name */
    private static final String __WORKSPACES_RESOURCE_CALENDAR_ROOT_NODE_NAME = "resource-calendar-root";
    
    private static final String __CALENDAR_CACHE_REQUEST_ATTR = CalendarWorkspaceModule.class.getName() + "$calendarCache";

    private static final String __EVENT_NUMBER_HEADER_ID = __WORKSPACES_CALENDARS_NODE_NAME + "$event_number";
    
    private CalendarDAO _calendarDAO;
    private CalendarEventJSONHelper _calendarEventJSONHelper;

    private I18nizableText _defaultCalendarTemplateDesc;
    private String _defaultCalendarColor;
    private String _defaultCalendarVisibility;
    private String _defaultCalendarWorkflowName;
    private I18nizableText _defaultCalendarTitle;
    private I18nizableText _defaultCalendarDescription;
    
    private I18nizableText _resourceCalendarTemplateDesc;
    private String _resourceCalendarColor;
    private String _resourceCalendarVisibility;
    private String _resourceCalendarWorkflowName;
    private I18nizableText _resourceCalendarTitle;
    private I18nizableText _resourceCalendarDescription;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _calendarDAO = (CalendarDAO) manager.lookup(CalendarDAO.ROLE);
        _calendarEventJSONHelper = (CalendarEventJSONHelper) manager.lookup(CalendarEventJSONHelper.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _defaultCalendarTemplateDesc = I18nizableText.parseI18nizableText(configuration.getChild("template-desc"), "plugin." + _pluginName, "");
        _defaultCalendarColor = configuration.getChild("color").getValue("col1");
        _defaultCalendarVisibility = configuration.getChild("visibility").getValue(CalendarVisibility.PRIVATE.name());
        _defaultCalendarWorkflowName = configuration.getChild("workflow").getValue("calendar-default");
        _defaultCalendarTitle = I18nizableText.parseI18nizableText(configuration.getChild("title"), "plugin." + _pluginName);
        _defaultCalendarDescription = I18nizableText.parseI18nizableText(configuration.getChild("description"), "plugin." + _pluginName, "");
        
        _resourceCalendarTemplateDesc = I18nizableText.parseI18nizableText(configuration.getChild("resource-template-desc"), "plugin." + _pluginName, "");
        _resourceCalendarColor = configuration.getChild("resource-color").getValue("resourcecol0");
        _resourceCalendarVisibility = configuration.getChild("resource-visibility").getValue(CalendarVisibility.PRIVATE.name());
        _resourceCalendarWorkflowName = configuration.getChild("resource-workflow").getValue("calendar-default");
        _resourceCalendarTitle = I18nizableText.parseI18nizableText(configuration.getChild("resource-title"), "plugin." + _pluginName);
        _resourceCalendarDescription = I18nizableText.parseI18nizableText(configuration.getChild("resource-description"), "plugin." + _pluginName, "");

    }
    
    @Override
    public String getId()
    {
        return CALENDAR_MODULE_ID;
    }
    
    public int getOrder()
    {
        return ORDER_CALENDAR;
    }
    
    public String getModuleName()
    {
        return __WORKSPACES_CALENDARS_NODE_NAME;
    }
    
    @Override
    protected String getModulePageName()
    {
        return "calendars";
    }
    
    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_CALENDARS_TITLE");
    }
    
    @Override
    protected void initializeModulePage(ModifiablePage calendarPage)
    {
        ModifiableZone defaultZone = calendarPage.createZone("default");
        
        String serviceId = "org.ametys.plugins.workspaces.module.Calendar";
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.SERVICE);
        defaultZoneItem.setServiceId(serviceId);
        
        ModifiableModelAwareDataHolder serviceDataHolder = defaultZoneItem.getServiceParameters();
        serviceDataHolder.setValue("xslt", _getDefaultXslt(serviceId));
    }
    
    /**
     * Get the calendars of a project
     * @param project The project
     * @return The list of calendar
     */
    public AmetysObjectIterable<Calendar> getCalendars(Project project)
    {
        ModifiableResourceCollection calendarRoot = getCalendarsRoot(project, true);
        return calendarRoot != null ? calendarRoot.getChildren() : null;
    }
    
    /**
     * Get the URI of a thread in project'site
     * @param project The project
     * @param calendarId The id of calendar
     * @param eventId The id of event
     * @return The thread uri
     */
    public String getEventUri(Project project, String calendarId, String eventId)
    {
        String moduleUrl = getModuleUrl(project);
        if (moduleUrl != null)
        {
            StringBuilder sb = new StringBuilder();
            sb.append(moduleUrl);
            
            try
            {
                CalendarEvent event = _resolver.resolveById(eventId);
                
                if (event.getStartDate() != null)
                {
                    DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
                    sb.append("?date=").append(df.format(DateUtils.asDate(event.getStartDate())));
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Nothing
            }
            
            sb.append("#").append(calendarId);
            
            return sb.toString();
        }
        
        return null;
    }
    
    /**
     * Add additional information on project and parent calendar
     * @param event The event
     * @param eventData The event data to complete
     */
    @SuppressWarnings("unchecked")
    protected void _addAdditionalEventData(CalendarEvent event, Map<String, Object> eventData)
    {
        Request request = ContextHelper.getRequest(_context);
        
        Calendar calendar = event.getParent();
        Project project = _projectManager.getParentProject(calendar);
        
        // Try to get calendar from cache if request is not null
        if (request.getAttribute(__CALENDAR_CACHE_REQUEST_ATTR) == null)
        {
            request.setAttribute(__CALENDAR_CACHE_REQUEST_ATTR, new HashMap<>());
        }
        
        Map<String, Object> calendarCache = (Map<String, Object>) request.getAttribute(__CALENDAR_CACHE_REQUEST_ATTR);
        
        if (!calendarCache.containsKey(calendar.getId()))
        {
            Map<String, Object> calendarInfo = new HashMap<>();
            
            calendarInfo.put("calendarName", calendar.getName());
            calendarInfo.put("calendarIsPublic", CalendarVisibility.PUBLIC.equals(calendar.getVisibility()));
            calendarInfo.put("calendarHasViewRight", canView(calendar));
            
            calendarInfo.put("projectId", project.getId());
            calendarInfo.put("projectTitle", project.getTitle());
            
            Set<Page> calendarModulePages = _projectManager.getModulePages(project, this);
            if (!calendarModulePages.isEmpty())
            {
                Page calendarModulePage = calendarModulePages.iterator().next();
                calendarInfo.put("calendarModulePageId", calendarModulePage.getId());
            }
            
            calendarCache.put(calendar.getId(), calendarInfo);
        }
        
        eventData.putAll((Map<String, Object>) calendarCache.get(calendar.getId()));
       
        eventData.put("eventUrl", getEventUri(project, calendar.getId(), event.getId()));
    }
    
    /**
     * Get the upcoming events of the calendars on which the user has a right
     * @param months the amount of months from today in which look for upcoming events
     * @param maxResults the maximum results to display
     * @param calendarIds the ids of the calendars to gather events from, null for all calendars
     * @param tagIds the ids of the valid tags for the events, null for any tag
     * @return the upcoming events
     */
    public List<Map<String, Object>> getUpcomingEvents(int months, int maxResults, List<String> calendarIds, List<String> tagIds)
    {
        List<Map<String, Object>> basicEventList = new ArrayList<> ();
        List<Map<String, Object>> recurrentEventList = new ArrayList<> ();
        
        java.util.Calendar cal = java.util.Calendar.getInstance();
        cal.set(java.util.Calendar.HOUR_OF_DAY, 0);
        cal.set(java.util.Calendar.MINUTE, 0);
        cal.set(java.util.Calendar.SECOND, 0);
        cal.set(java.util.Calendar.MILLISECOND, 0);
        ZonedDateTime startDate = ZonedDateTime.now().truncatedTo(ChronoUnit.DAYS);

        ZonedDateTime endDate = startDate.plusMonths(months);
        
        Expression nonRecurrentExpr = new StringExpression(JCRCalendarEvent.ATTRIBUTE_RECURRENCE_TYPE, Operator.EQ, "NEVER");
        Expression startDateExpr = new DateExpression(JCRCalendarEvent.ATTRIBUTE_START_DATE, Operator.GE, DateUtils.asDate(startDate));
        Expression endDateExpr = new DateExpression(JCRCalendarEvent.ATTRIBUTE_START_DATE, Operator.LE, DateUtils.asDate(endDate));
        
        Expression keywordsExpr = null;
        
        if (tagIds != null && !tagIds.isEmpty())
        {
            List<Expression> orExpr = new ArrayList<>();
            for (String tagId : tagIds)
            {
                orExpr.add(new StringExpression(JCRCalendarEvent.ATTRIBUTE_KEYWORDS, Operator.EQ, tagId));
            }
            keywordsExpr = new OrExpression(orExpr.toArray(new Expression[orExpr.size()]));
        }
        
        // Get the non recurrent events sorted by ascending date and within the configured range
        Expression eventExpr = new AndExpression(nonRecurrentExpr, startDateExpr, endDateExpr, keywordsExpr);
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(JCRCalendarEvent.ATTRIBUTE_START_DATE, true, false);
        
        String basicEventQuery = QueryHelper.getXPathQuery(null, JCRCalendarEventFactory.CALENDAR_EVENT_NODETYPE, eventExpr, sortCriteria);
        AmetysObjectIterable<CalendarEvent> basicEvents = _resolver.query(basicEventQuery);
        AmetysObjectIterator<CalendarEvent> basicEventIt = basicEvents.iterator();
        
        int processed = 0;
        while (basicEventIt.hasNext() && processed < maxResults)
        {
            CalendarEvent event = basicEventIt.next();
            Calendar holdingCalendar = (Calendar) event.getParent();
            
            if (_filterEvent(calendarIds, event) && _hasAccess(holdingCalendar))
            {
                // The event is in the list of selected calendars and has the appropriate tags (can be none if tagIds == null)
                
                // FIXME should use something like an EventInfo object with some data + calendar, project name
                // And use a function to process the transformation...
                // Function<EventInfo, Map<String, Object>> fn. eventData.putAll(fn.apply(info));
                
                // standard set of event data
                Map<String, Object> eventData = _calendarEventJSONHelper.eventAsJson(event, false, false);
                basicEventList.add(eventData);
                processed++;
                
                // add additional info
                _addAdditionalEventData(event, eventData);
            }
        }
        
        Expression recurrentExpr = new StringExpression(JCRCalendarEvent.ATTRIBUTE_RECURRENCE_TYPE, Operator.NE, "NEVER");
        eventExpr = new AndExpression(recurrentExpr, keywordsExpr);
        
        String recurrentEventQuery = QueryHelper.getXPathQuery(null, JCRCalendarEventFactory.CALENDAR_EVENT_NODETYPE, eventExpr, sortCriteria);
        AmetysObjectIterable<CalendarEvent> recurrentEvents = _resolver.query(recurrentEventQuery);
        AmetysObjectIterator<CalendarEvent> recurrentEventIt = recurrentEvents.iterator();
        
        // FIXME cannot count processed here...
        processed = 0;
        while (recurrentEventIt.hasNext() /*&& processed < maxResultsAsInt*/)
        {
            CalendarEvent event = recurrentEventIt.next();
            Optional<CalendarEventOccurrence> nextOccurrence = event.getNextOccurrence(new CalendarEventOccurrence(event, startDate));
            
            // The recurrent event first occurrence is within the range
            if (nextOccurrence.isPresent() && nextOccurrence.get().before(endDate))
            {
                // FIXME calculate occurrences only if keep event...
                List<CalendarEventOccurrence> occurrences = event.getOccurrences(nextOccurrence.get().getStartDate(), endDate);
                Calendar holdingCalendar = (Calendar) event.getParent();
                
                if (_filterEvent(calendarIds, event) && _hasAccess(holdingCalendar))
                {
                    // The event is in the list of selected calendars and has the appropriate tags (can be none if tagIds == null)
                    
                    // Add all its occurrences that are within the range
                    for (CalendarEventOccurrence occurrence : occurrences)
                    {
                        Map<String, Object> eventData = _calendarEventJSONHelper.eventAsJsonWithOccurrence(event, occurrence.getStartDate(), false);
                        recurrentEventList.add(eventData);
                        processed++;
                        
                        _addAdditionalEventData(event, eventData);
                        
                    }
                }
            }
        }
        
        // Re-sort chronologically the events' union
        List<Map<String, Object>> allEvents = ListUtils.union(basicEventList, recurrentEventList);
        Collections.sort(allEvents, new StartDateComparator());

        // Return the first maxResults events
        return allEvents.size() <= maxResults ? allEvents : allEvents.subList(0, maxResults);
    }
    
    /**
     * Determine whether the given event has to be kept or not depending on the given calendars
     * @param calendarIds the ids of the calendars
     * @param event the event
     * @return true if the event can be kept, false otherwise
     */
    private boolean _filterEvent(List<String> calendarIds, CalendarEvent event)
    {
        Calendar holdingCalendar = (Calendar) event.getParent();
        // FIXME calendarIds.get(0) == null means "All calendars" selected in the select calendar widget ??
        // need cleaner code
        return calendarIds == null || calendarIds.get(0) == null || calendarIds.contains(holdingCalendar.getId());
    }
    
    private boolean _hasAccess(Calendar calendar)
    {
        return CalendarVisibility.PUBLIC.equals(calendar.getVisibility()) || canView(calendar);
    }
        
    /**
     * Indicates if the current user can view the calendar
     * @param calendar The calendar to test
     * @return true if the calendar can be viewed
     */
    public boolean canView(Calendar calendar)
    {
        return _rightManager.currentUserHasReadAccess(calendar);
    }
    
    /**
     * Indicates if the current user can view the event
     * @param event The event to test
     * @return true if the event can be viewed
     */
    public boolean canView(CalendarEvent event)
    {
        return _rightManager.currentUserHasReadAccess(event.getParent());
    }
    
    /**
     * Compares events on their starting date
     */
    protected static class StartDateComparator implements Comparator<Map<String, Object>>
    {
        @Override
        public int compare(Map<String, Object> calendarEventInfo1, Map<String, Object> calendarEventInfo2)
        {
            String startDate1asString = (String) calendarEventInfo1.get("startDate");
            String startDate2asString = (String) calendarEventInfo2.get("startDate");
            
            Date startDate1 = DateUtils.parse(startDate1asString);
            Date startDate2 = DateUtils.parse(startDate2asString);
            
            // The start date is before if
            return startDate1.compareTo(startDate2);
        }
    }
    
    @Override
    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of("calendar.event.created", "calendar.event.updated", "calendar.event.deleting");
    }

    @Override
    protected void _internalActivateModule(Project project, Map<String, Object> additionalValues)
    {
        createResourceCalendar(project, additionalValues);
        _createDefaultCalendar(project, additionalValues);
    }

    /**
     * Create a calendar to store resources if needed
     * @param project the project
     * @param additionalValues A list of optional additional values. Accepted values are : description, mailingList, inscriptionStatus, defaultProfile, tags, categoryTags, keywords and language
     * @return The resource calendar
     */
    public Calendar createResourceCalendar(Project project, Map<String, Object> additionalValues)
    {
        ModifiableResourceCollection resourceCalendarRoot = getResourceCalendarRoot(project, true);
        
        String lang;
        if (additionalValues.containsKey("language"))
        {
            lang = (String) additionalValues.get("language");
        }
        else
        {
            lang = getLang(project);
        }

        Calendar resourceCalendar = resourceCalendarRoot.getChildren()
                .stream()
                .filter(Calendar.class::isInstance)
                .map(Calendar.class::cast)
                .findFirst()
                .orElse(null);
        
        if (resourceCalendar == null)
        {
            Boolean renameIfExists = false;
            Boolean checkRights = false;
            String description = _i18nUtils.translate(_resourceCalendarDescription, lang);
            String inputName = _i18nUtils.translate(_resourceCalendarTitle, lang);
            String templateDesc = _i18nUtils.translate(_resourceCalendarTemplateDesc, lang);
            try
            {
                Map result = _calendarDAO.addCalendar(resourceCalendarRoot, inputName, description, templateDesc, _resourceCalendarColor, _resourceCalendarVisibility, _resourceCalendarWorkflowName, renameIfExists, checkRights, false);

                resourceCalendar = _resolver.resolveById((String) result.get("id"));
            }
            catch (Exception e)
            {
                getLogger().error("Error while trying to create the first calendar in a newly created project", e);
            }
        }
        return resourceCalendar;
    }
    
    private void _createDefaultCalendar(Project project, Map<String, Object> additionalValues)
    {
        ModifiableResourceCollection moduleRoot = getCalendarsRoot(project, true);
        
        if (moduleRoot != null && !_hasOtherCalendar(project))
        {
            Boolean renameIfExists = false;
            Boolean checkRights = false;

            String lang;
            if (additionalValues.containsKey("language"))
            {
                lang = (String) additionalValues.get("language");
            }
            else
            {
                lang = getLang(project);
            }
            
            String description = _i18nUtils.translate(_defaultCalendarDescription, lang);
            String inputName = _i18nUtils.translate(_defaultCalendarTitle, lang);
            String templateDesc = _i18nUtils.translate(_defaultCalendarTemplateDesc, lang);
            try
            {
                _calendarDAO.addCalendar(moduleRoot, inputName, description, templateDesc, _defaultCalendarColor, _defaultCalendarVisibility, _defaultCalendarWorkflowName, renameIfExists, checkRights, false);
            }
            catch (Exception e)
            {
                getLogger().error("Error while trying to create the first calendar in a newly created project", e);
            }
            
        }
    }
    
    private boolean _hasOtherCalendar(Project project)
    {
        AmetysObjectIterable<Calendar> calendars = getCalendars(project);
        
        return calendars != null && calendars.getSize() > 0;
    }
    
    /**
     * Get the lang for this project, or fr if none found
     * @param project the project to check
     * @return a lang in this project
     */
    protected String getLang(Project project)
    {
        /*
         * get the site, then the sitemaps (transformed back to a stream of sitemap)
         * then get the name
         * return the 1st one, or "en" by default if none is available, or if the site could not be found
         */
        
        Site site = project.getSite();
        
        if (site != null)
        {
            return site.getSitemaps()
                       .stream()
                       .filter(Objects::nonNull)
                       .map(Sitemap::getName)
                       .findFirst()
                       .orElse("en");
        }
        else
        {
            return "en";
        }
    }

    /**
     * Get the calendars of a project
     * @param project The project
     * @return The list of calendar
     */
    public Calendar getResourceCalendar(Project project)
    {
        ModifiableResourceCollection resourceCalendarRoot = getResourceCalendarRoot(project, true);
        return resourceCalendarRoot.getChildren()
                .stream()
                .filter(Calendar.class::isInstance)
                .map(Calendar.class::cast)
                .findFirst()
                .orElse(createResourceCalendar(project, new HashMap<>()));
    }

    /**
     * Get the root for tasks
     * @param project The project
     * @param create true to create root if not exists
     * @return The root for tasks
     */
    public ModifiableResourceCollection getCalendarResourcesRoot(Project project, boolean create)
    {
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, create);
        return _getAmetysObject(moduleRoot, __WORKSPACES_CALENDAR_RESOURCES_ROOT_NODE_NAME, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE, create);
    }

    /**
     * Get the root for tasks
     * @param project The project
     * @param create true to create root if not exists
     * @return The root for tasks
     */
    public ModifiableResourceCollection getCalendarsRoot(Project project, boolean create)
    {
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, create);
        return _getAmetysObject(moduleRoot, __WORKSPACES_CALENDARS_ROOT_NODE_NAME, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE, create);
    }

    /**
     * Get the root for tasks
     * @param project The project
     * @param create true to create root if not exists
     * @return The root for tasks
     */
    public ModifiableResourceCollection getResourceCalendarRoot(Project project, boolean create)
    {
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, create);
        return _getAmetysObject(moduleRoot, __WORKSPACES_RESOURCE_CALENDAR_ROOT_NODE_NAME, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE, create);
    }
    
    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    {
        if (isActive)
        {
            AmetysObjectIterable<Calendar> calendars = getCalendars(project);
            Calendar ressourceCalendar = getResourceCalendar(project);
            
            // concatenate both type of calendars
            long eventNumber = Stream.concat(calendars.stream(), Stream.of(ressourceCalendar))
                    // get all events for each calendar (events are children of calendar node)
                    .map(TraversableAmetysObject::getChildren)
                    // use flatMap to have a stream with all events from all calendars
                    .flatMap(AmetysObjectIterable::stream)
                    // count the number of events
                    .count();
            
            return Map.of(__EVENT_NUMBER_HEADER_ID, eventNumber);
        }
        else
        {
            return Map.of(__EVENT_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }

    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__EVENT_NUMBER_HEADER_ID, new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_EVENT_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }

    @Override
    public Set<String> getAllEventTypes()
    {
        return Set.of(ObservationConstants.EVENT_CALENDAR_CREATED,
                      ObservationConstants.EVENT_CALENDAR_DELETED,
                      ObservationConstants.EVENT_CALENDAR_EVENT_CREATED,
                      ObservationConstants.EVENT_CALENDAR_EVENT_DELETING,
                      ObservationConstants.EVENT_CALENDAR_EVENT_UPDATED,
                      ObservationConstants.EVENT_CALENDAR_MOVED,
                      ObservationConstants.EVENT_CALENDAR_RESOURCE_CREATED,
                      ObservationConstants.EVENT_CALENDAR_RESOURCE_DELETED,
                      ObservationConstants.EVENT_CALENDAR_RESOURCE_UPDATED,
                      ObservationConstants.EVENT_CALENDAR_UPDATED);
    }
}

