/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.categories;

import java.util.HashMap;
import java.util.Map;

import org.ametys.cms.color.AbstractColorsComponent;
import org.ametys.cms.tag.ColorableTag;
import org.ametys.cms.tag.Tag;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Class representing a category. <br>
 */
public class Category implements ColorableTag
{
    private String _id;
    private String _name;
    private I18nizableText _title;
    private I18nizableText _description;
    private String _color;
    private AbstractColorsComponent _colorsComponent;
    private Map<String, Category> _tags;
    private Category _parent;

    /**
     * Constructor
     * @param id The id of the tag. The id must be unique.
     */
    public Category(String id)
    {
        _id = id;
    }
    
    /**
     * Constructor
     * @param id The id of the tag. The id must be unique.
     * @param name The name of the tag. The name must be unique. The name is the same as the id except for JCR tag.
     * @param parent The parent tag (can be null if the tag has no parent)
     * @param title the tag title
     * @param description the tag description
     */
    public Category(String id, String name, Category parent, I18nizableText title, I18nizableText description)
    {
        _id = id;
        _name = name;
        _parent = parent;
        _title = title;
        _description = description;
    }
    
    /**
     * Constructor
     * @param id The id of the tag. The id must be unique.
     * @param name The name of the tag. The name must be unique. The name is the same as the id except for JCR tag.
     * @param color the color
     * @param colorsComponent the colors component
     * @param parent The parent tag (can be null if the tag has no parent)
     * @param title the tag title
     * @param description the tag description
     */
    public Category(String id, String name, String color, AbstractColorsComponent colorsComponent, Category parent, I18nizableText title, I18nizableText description)
    {
        this(id, name, parent, title, description);
        _color = color;
        _colorsComponent = colorsComponent;
    }
    
    @Override
    public String getId ()
    {
        return _id;
    }
    
    @Override
    public String getName ()
    {
        return _name;
    }
    
    @Override
    public I18nizableText getTitle()
    {
        return _title;
    }
    
    @Override
    public I18nizableText getDescription()
    {
        return _description;
    }
    
    @Override
    public String getColor(boolean withDefaultValue)
    {
        return _color;
    }
    
    @Override
    public AbstractColorsComponent getColorComponent()
    {
        return _colorsComponent;
    }
    
    @Override
    public String getParentName()
    {
        return _parent != null ? _parent.getName() : null;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Category getParent()
    {
        return _parent;
    }
    
    @Override
    public void addTag(Tag tag)
    {
        if (_tags == null)
        {
            _tags = new HashMap<>();
        }
        
        _tags.put(tag.getId(), (Category) tag);
    }
    
    @Override
    public Map<String, Category> getTags()
    {
        if (_tags == null)
        {
            _tags = new HashMap<>();
        }
        
        return _tags;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Category getTag(String tagId)
    {
        if (_tags == null)
        {
            _tags = new HashMap<>();
        }
        
        return _tags.get(tagId);
    }
    
    @Override
    public boolean hasTag(String tagId)
    {
        if (_tags == null)
        {
            _tags = new HashMap<>();
        }
        
        return _tags.containsKey(tagId);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public void setTags(Map<String, ? extends Tag> tags)
    {
        _tags = (Map<String, Category>) tags;
    }
}
