/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.chat;

import java.util.Map;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.runtime.config.Config;

/**
 * When a member content is validated, the chat server need to be updated (email, name, avatar...)
 */
public class MemberValidatedObserver extends AbstractLogEnabled implements Observer, Serviceable // Needs to be synchronous otherwise, there is no current site when updating avatar url that will lead to cocoon://null/_contents url
{
    /** Rocket.Chat helper */
    protected ChatHelper _chatHelper;
    
    private ContentTypesHelper _contentTypesHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _chatHelper = (ChatHelper) manager.lookup(ChatHelper.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }

    public boolean supports(Event event)
    {
        if (!ObservationConstants.EVENT_CONTENT_VALIDATED.equals(event.getId()) || !Config.getInstance().<Boolean>getValue("workspaces.chat.active"))
        {
            return false;
        }
        
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        return _contentTypesHelper.isInstanceOf(content, WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
    }

    public int getPriority(Event event)
    {
        return MAX_PRIORITY; // Yes, "MAX PRIORITY" is the lowest priority...
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        UserIdentity user = content.getValue("user");
        if (user != null)
        {
            try 
            {
                _chatHelper.updateUserInfos(user, false);
            }
            catch (Exception e)
            {
                // Only debug since user may not exist due to inconsistency between servers
                getLogger().debug("An error occurred while update userinfo on chat server for " + UserIdentity.userIdentityToString(user), e);
            }
        }
    }
}
