/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.chat;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.Base64;
import java.util.Base64.Encoder;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Redirect to the video server with user informations
 */
public class VideoRedirectionAction extends ServiceableAction implements Initializable, ThreadSafe
{
    private CurrentUserProvider _currentUserProvider;
    private UserManager _userManager;
    private MessageDigest _digest;
    private I18nUtils _i18nUtils;
    private Encoder _base64Encoder;
    private ProjectMemberManager _projectMemberManager;
    private ProjectManager _projectManager;
    private UserDirectoryHelper _userDirectoryHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) smanager.lookup(ProjectMemberManager.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _digest = MessageDigest.getInstance("SHA-256");
        _base64Encoder = Base64.getUrlEncoder().withoutPadding();
        
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        String videoType = parameters.getParameter("type", "room");
        String target = parameters.getParameter("target", null);
        
        if (userIdentity != null && target != null)
        {
            User user = _userManager.getUser(userIdentity);
            
            _checkRights(videoType, target, user);
            
            String url = Config.getInstance().getValue("workspaces.chat.video.url")
                        + "/"
                        + _hash(videoType, target)
                        + "/"
                        + URIUtils.encodePathSegment(_readable(videoType, target))
                        + "#"
                        + URIUtils.encodeFragment(_config(user));
            
            redirector.globalRedirect(false, url);
        }
        
        return null;
    }

    private void _checkRights(String videoType, String target, User user)
    {
        if ("room".equals(videoType))
        {
            Project project = _projectManager.getProject(target);
            if (project == null)
            {
                throw new IllegalStateException("There is no project named " + target);
            }
            if (!_projectMemberManager.isProjectMember(project, user.getIdentity()))
            {
                throw new AccessDeniedException("User " + UserIdentity.userIdentityToString(user.getIdentity()) + " cannot connect to the video of project " + target);
            }
        }
    }

    private String _readable(String videoType, String target)
    {
        if ("room".equals(videoType))
        {
            // target is the room name
            return target;
        }
        else
        {
            // target is the interlocutors
            return _i18nUtils.translate(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CHAT_VIDEO_DIRECT_LINK_TITLE"));
        }
    }
    
    private String _hash(String videoType, String target)
    {
        String tohash;
        
        if ("room".equals(videoType))
        {
            // target is the room name
            tohash = target;
        }
        else
        {
            // target is the interlocutors
            tohash = target;
        }
        
        return _base64Encoder.encodeToString(_digest.digest(tohash.getBytes(StandardCharsets.UTF_8)));
    }

    private String _config(User user)
    {
        String userFullname = user.getFullName();
        String userEmail = user.getEmail();
        Content userContent = _userDirectoryHelper.getUserContent(user.getIdentity(), null);
        if (userContent != null)
        {
            userFullname = StringUtils.defaultIfBlank(StringUtils.join(new String[] {userContent.getValue("firstname"), userContent.getValue("lastname")}, " "), userFullname);
            userEmail = StringUtils.defaultIfBlank(userContent.getValue("email"), userEmail);
        }
        

        
        return "userInfo.displayName=\"" + userFullname + "\"" 
                + "&userInfo.email=\"" + userEmail + "\"" 
                + "&config.disableInviteFunctions=true" 
                + "&config.prejoinPageEnabled=false" 
                + "&config.requireDisplayName=false";
    }

}
