/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.cmis;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.Map;

import org.apache.chemistry.opencmis.commons.definitions.MutableDocumentTypeDefinition;
import org.apache.chemistry.opencmis.commons.definitions.MutableFolderTypeDefinition;
import org.apache.chemistry.opencmis.commons.definitions.MutablePropertyDefinition;
import org.apache.chemistry.opencmis.commons.definitions.MutableTypeDefinition;
import org.apache.chemistry.opencmis.commons.definitions.PropertyDefinition;
import org.apache.chemistry.opencmis.commons.definitions.TypeDefinition;
import org.apache.chemistry.opencmis.commons.definitions.TypeDefinitionList;
import org.apache.chemistry.opencmis.commons.enums.CmisVersion;
import org.apache.chemistry.opencmis.commons.exceptions.CmisObjectNotFoundException;
import org.apache.chemistry.opencmis.commons.server.CallContext;
import org.apache.chemistry.opencmis.server.support.TypeDefinitionFactory;

/**
 * Manages the type definitions for all FileShare repositories.
 */
public class CmisTypeManager
{
    private static final String NAMESPACE = "http://ametysorg/opencmis";

    private final TypeDefinitionFactory _typeDefinitionFactory;
    private final Map<String, TypeDefinition> _typeDefinitions;

    /**
     * Type Manager
     */
    public CmisTypeManager()
    {
        // set up TypeDefinitionFactory
        _typeDefinitionFactory = TypeDefinitionFactory.newInstance();
        _typeDefinitionFactory.setDefaultNamespace(NAMESPACE);
        _typeDefinitionFactory.setDefaultControllableAcl(false);
        _typeDefinitionFactory.setDefaultControllablePolicy(false);
        _typeDefinitionFactory.setDefaultQueryable(false);
        _typeDefinitionFactory.setDefaultFulltextIndexed(false);
        _typeDefinitionFactory.setDefaultTypeMutability(_typeDefinitionFactory.createTypeMutability(false, false, false));

        // set up definitions map
        _typeDefinitions = new HashMap<>();

        // add base folder type
        MutableFolderTypeDefinition folderType = _typeDefinitionFactory
                .createBaseFolderTypeDefinition(CmisVersion.CMIS_1_1);
        removeQueryableAndOrderableFlags(folderType);
        _typeDefinitions.put(folderType.getId(), folderType);

        // add base document type
        MutableDocumentTypeDefinition documentType = _typeDefinitionFactory
                .createBaseDocumentTypeDefinition(CmisVersion.CMIS_1_1);
        removeQueryableAndOrderableFlags(documentType);
        _typeDefinitions.put(documentType.getId(), documentType);
    }

    /*
     * Removes the queryable and orderable flags from the property definitions
     * of a type definition because this implementations does neither support
     * queries nor can order objects.
     */
    private void removeQueryableAndOrderableFlags(MutableTypeDefinition type)
    {
        for (PropertyDefinition<?> propDef : type.getPropertyDefinitions().values())
        {
            MutablePropertyDefinition<?> mutablePropDef = (MutablePropertyDefinition<?>) propDef;
            mutablePropDef.setIsQueryable(false);
            mutablePropDef.setIsOrderable(false);
        }
    }

    /**
     * Returns the internal type definition.
     * @param typeId String
     * @return Type definition
     */
    public synchronized TypeDefinition getInternalTypeDefinition(String typeId)
    {
        return _typeDefinitions.get(typeId);
    }

    // --- service methods ---

    /**
     * Get the type definition
     * @param context call context
     * @param typeId type id
     * @return type definition
     */
    public TypeDefinition getTypeDefinition(CallContext context, String typeId)
    {
        TypeDefinition type = _typeDefinitions.get(typeId);
        if (type == null)
        {
            throw new CmisObjectNotFoundException("Type '" + typeId + "' is unknown!");
        }

        return _typeDefinitionFactory.copy(type, true, context.getCmisVersion());
    }

    /**
     * get the type children
     * @param context call context
     * @param typeId type
     * @param includePropertyDefinitions excludes
     * @param maxItems max
     * @param skipCount skip
     * @return list of type definition
     */
    public TypeDefinitionList getTypeChildren(CallContext context, String typeId, Boolean includePropertyDefinitions,
            BigInteger maxItems, BigInteger skipCount)
    {
        return _typeDefinitionFactory.createTypeDefinitionList(_typeDefinitions, typeId, includePropertyDefinitions,
                maxItems, skipCount, context.getCmisVersion());
    }

    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder();

        for (TypeDefinition type : _typeDefinitions.values())
        {
            sb.append('[');
            sb.append(type.getId());
            sb.append(" (");
            sb.append(type.getBaseTypeId().value());
            sb.append(")]");
        }

        return sb.toString();
    }
}
