/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.util.Collections;

import org.ametys.cms.transformation.URIResolver;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.editor.ResourceURIResolver;

/**
 * {@link URIResolver} for type "project-resource". <br>
 * These links point to a file from the resources of a project.
 */
public class ProjectResourceURIResolver extends ResourceURIResolver implements PluginAware
{
    private String _pluginName;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public String getType()
    {
        return "project-resource";
    }
    
    @Override
    public I18nizableText getLabel(String uri)
    {
        try
        {
            Resource resource = (Resource) _resolver.resolveById(uri);
            return new I18nizableText("plugin" + _pluginName, "PLUGINS_WORKSPACES_LINK_RESOURCE_LABEL", Collections.singletonList(resource.getResourcePath()));
        }
        catch (UnknownAmetysObjectException e)
        {
            return new I18nizableText("plugin" + _pluginName, "PLUGINS_WORKSPACES_LINK_RESOURCE_UNKNOWN");
        }
    }

    @Override
    protected String getUriPrefix (AmetysObject object, boolean download, boolean absolute, boolean internal)
    {
        String prefix = super.getUriPrefix(object, download, absolute, internal);
        return prefix + "/_plugins/" + _pluginName + "/" + _getProject(object).getName();
    }
    
    @Override
    protected String _getSiteName(Resource resource)
    {
        Project project = _getProject(resource);
        
        return project.getSite().getName();
    }
    
    /**
     * Retrieves parent project
     * @param resource The resource which belongs to a project
     * @return The parent project
     */
    protected Project _getProject(AmetysObject resource)
    {
        AmetysObject parent = resource.getParent();
        while (parent != null)
        {
            if (parent instanceof Project)
            {
                return (Project) parent;
            }
            
            parent = parent.getParent();
        }
        
        return null;
    }
}
