/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.indexing.solr;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.solr.client.solrj.SolrClient;

import org.ametys.cms.indexing.IndexingException;
import org.ametys.cms.indexing.solr.DocumentProvider;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * {@link DocumentProvider} for indexing documents (project resources...) during workspace indexation
 */
public class ProjectDocumentProvider extends AbstractLogEnabled implements DocumentProvider, Serviceable, Contextualizable
{
    private static final I18nizableText __LABEL = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CONTENT_INDEXING_PROJETDOCUMENTPROVIDER"); 
    /** The manager for {@link Project}s */
    protected ProjectManager _projectManager;
    /** The Solr indexer for project resources */
    protected SolrProjectResourceIndexer _solrProjectResourceIndexer;
    /** The manager for documents */
    protected DocumentWorkspaceModule _documentModule;
    
    /** The component context. */
    protected Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _solrProjectResourceIndexer = (SolrProjectResourceIndexer) manager.lookup(SolrProjectResourceIndexer.ROLE);
        WorkspaceModuleExtensionPoint moduleManager = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _documentModule = moduleManager.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
    }
    
    public I18nizableText getLabel()
    {
        return __LABEL;
    }
    
    public void indexDocuments(String workspaceName, SolrClient solrClient) throws IndexingException
    {
        indexDocuments(workspaceName, solrClient, ProgressionTrackerFactory.createContainerProgressionTracker("Index documents", getLogger()));
    }
    
    public void indexDocuments(String workspaceName, SolrClient solrClient, ContainerProgressionTracker progressionTracker) throws IndexingException
    {
        Request request = ContextHelper.getRequest(_context);
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, workspaceName);
            
            _indexProjects(workspaceName, solrClient, progressionTracker);
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    private void _indexProjects(String workspaceName, SolrClient solrClient, ContainerProgressionTracker progressionTracker) throws IndexingException
    {
        AmetysObjectIterable<Project> projects = _projectManager.getProjects();

        
        if (projects.getSize() == 0)
        {
            progressionTracker.addSimpleStep("noprojects", new I18nizableText("")).setSize(0);
        }
        else
        {
            for (Project project : projects)
            {
                progressionTracker.addSimpleStep(project.getName(), new I18nizableText(project.getName()));
            }
            
            for (Project project : projects)
            {
                _indexProject(workspaceName, project, solrClient, progressionTracker.getStep(project.getName()));
            }
        }
    }
    
    private void _indexProject(String workspaceName, Project project, SolrClient solrClient, SimpleProgressionTracker progressionTracker) throws IndexingException
    {
        try
        {
            // Index the documents
            ModifiableResourceCollection documentRoot = _documentModule.getModuleRoot(project, false);
            getLogger().info("Starting indexation of project resources for project {} ('{}') in workspace {}", project.getName(), project.getId(), workspaceName);
            
            _solrProjectResourceIndexer.indexProjectResources(documentRoot, project, solrClient, progressionTracker);
        }
        catch (Exception e)
        {
            String error = String.format("Failed to index project %s in workspace %s", project.getName(), workspaceName);
            getLogger().error(error, e);
            throw new IndexingException(error, e);
        }
    }
}
