/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members.observers;

import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.userdirectory.page.UserDirectoryPageResolver;
import org.ametys.plugins.userdirectory.page.UserPage;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * {@link Observer} to invalidate the zoneitem containing the member' content when a member was added or removed from a project
 *
 */
public class InvalidateZoneItemCacheOnMemberUpdatedObserver extends AbstractMemberObserver
{
    private PageElementCache _zoneItemCache;
    private UserDirectoryPageHandler _udPageHandler;
    private UserDirectoryPageResolver _udPageResolver;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _zoneItemCache = (PageElementCache) smanager.lookup(PageElementCache.ROLE + "/zoneItem");
        _udPageHandler = (UserDirectoryPageHandler) smanager.lookup(UserDirectoryPageHandler.ROLE);
        _udPageResolver = (UserDirectoryPageResolver) smanager.lookup(UserDirectoryPageResolver.ROLE);
    }
    
    @Override
    protected void _internalObserve(Event event, List<Content> userContents) throws Exception
    {
        for (Content content : userContents)
        {
            if (content instanceof JCRAmetysObject)
            {
                Set<Page> rootPages = _udPageHandler.getUserDirectoryRootPages(WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
                
                for (Page rootPage : rootPages)
                {
                    UserPage userPage = _udPageResolver.getUserPage(rootPage, content);
                    if (userPage != null)
                    {
                        // Get the zones of the page
                        AmetysObjectIterable< ? extends Zone> zones = userPage.getZones();
                        for (Zone zone : zones)
                        {
                            // For every zone item of every zone
                            for (ZoneItem zoneItem : zone.getZoneItems())
                            {
                                if (zoneItem.getType().equals(ZoneType.CONTENT)) 
                                {
                                    // Invalidate cache for all workspaces
                                    _zoneItemCache.removeItem(null, rootPage.getSiteName(), "CONTENT", zoneItem.getId());
                                }
                            }
                        }
                    }
                    
                }
            }
        }
    }
}
