/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.workspaces.activities.forums.ThreadActivityType;
import org.ametys.plugins.workspaces.activities.forums.ThreadCommentActivityType;
import org.ametys.plugins.workspaces.activities.forums.ThreadDeletedActivityType;
import org.ametys.plugins.workspaces.forum.ForumWorkspaceModule;
import org.ametys.plugins.workspaces.forum.jcr.JCRThread;
import org.ametys.plugins.workspaces.forum.json.ThreadJSONHelper;

/**
 * Generator for thread mail notification
 */
public class ForumMailNotifierGenerator extends AbstractMailNotifierGenerator
{
    private AmetysObjectResolver _resolver;
    private ThreadJSONHelper _threadJSONHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _threadJSONHelper = (ThreadJSONHelper) manager.lookup(ThreadJSONHelper.ROLE);
    }
    
    @Override
    protected void saxAdditionalData(Activity activity) throws SAXException
    {
        try
        {
            JCRThread thread = _resolver.resolveById(activity.getValue(ThreadActivityType.THREAD_ID));
            _saxComments(thread, activity);
            XMLUtils.createElement(contentHandler, "modulePageId", thread.getId());
            if (activity.getActivityType() instanceof ThreadDeletedActivityType)
            {
                XMLUtils.createElement(contentHandler, "deleted");
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            // nothing to do. If the thread doesn't exist anymore then we just don't provide additional info
        }
    }

    private void _saxComments(JCRThread thread, Activity currentActivity) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "comments");
        
        String currentCommentId = currentActivity.getValue(ThreadCommentActivityType.COMMENT_ID);
        for (RichTextComment comment : thread.getFlattenComments(true, true))
        {
            AttributesImpl attr = new AttributesImpl();
            if (comment.getId().equals(currentCommentId))
            {
                attr.addCDATAAttribute("currentActivity", "true");
            }
            XMLUtils.startElement(contentHandler, "comment", attr);
            
            XMLUtils.createElement(contentHandler, "authorName", comment.getAuthorName());
            
            UserIdentity userIdentity = comment.getAuthor();
            
            if (userIdentity != null)
            {
                XMLUtils.createElement(contentHandler, "login", userIdentity.getLogin());
                XMLUtils.createElement(contentHandler, "populationId", userIdentity.getPopulationId());
            }
            
            XMLUtils.createElement(contentHandler, "creationDate", DateUtils.zonedDateTimeToString(comment.getCreationDate()));
            XMLUtils.createElement(contentHandler, ThreadCommentActivityType.CONTENT, _threadJSONHelper.richTextToSimpleText(comment.getRichTextContent(), 150));
           
            XMLUtils.endElement(contentHandler, "comment");
        }
        XMLUtils.endElement(contentHandler, "comments");
    }

    @Override
    protected String _getModuleId()
    {
        return ForumWorkspaceModule.FORUM_MODULE_ID;
    }

}
