/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.tag.AbstractTagProviderExtensionPoint;
import org.ametys.cms.tag.DefaultTag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.LambdaUtils;
import org.ametys.core.util.LambdaUtils.ThrowingConsumer;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO;
import org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.objects.Project.InscriptionStatus;
import org.ametys.plugins.workspaces.tags.ProjectTagProviderExtensionPoint;
import org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;

/**
 * Generator used to gather informations on all of the projects of the application for the
 * projects' catalogue service
 */
public class SimpleProjectsCatalogueGenerator extends ServiceableGenerator
{
    /** The project manager component */
    private ProjectManager _projectManager;

    /** The project member manager */
    private ProjectMemberManager _projectMemberManager;

    /** The user manager */
    private UserManager _userManager;

    /** The current user provider */
    private CurrentUserProvider _currentUserProvider;

    private UserHelper _userHelper;

    /** The documents module DAO */
    private WorkspaceExplorerResourceDAO _workspaceExplorerResourceDAO;

    /** The tasks module DAO */
    private WorkspaceTaskDAO _workspaceTaskDAO;

    /** Workspace threads DAO */
    private WorkspaceThreadDAO _workspaceThreadDAO;

    private CategoryProviderExtensionPoint _categoryProviderEP;
    private ProjectTagProviderExtensionPoint _projectTagProviderEP;
    private CategoryHelper _categoryHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);

        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) serviceManager.lookup(ProjectMemberManager.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _workspaceExplorerResourceDAO = (WorkspaceExplorerResourceDAO) serviceManager.lookup(WorkspaceExplorerResourceDAO.ROLE);
        _workspaceTaskDAO = (WorkspaceTaskDAO) serviceManager.lookup(WorkspaceTaskDAO.ROLE);
        _workspaceThreadDAO = (WorkspaceThreadDAO) serviceManager.lookup(WorkspaceThreadDAO.ROLE);
        _projectTagProviderEP = (ProjectTagProviderExtensionPoint) manager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) serviceManager.lookup(CategoryProviderExtensionPoint.ROLE);
        _categoryHelper = (CategoryHelper) serviceManager.lookup(CategoryHelper.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        int max = parameters.getParameterAsInteger("max-results", 0);

        Request request = ObjectModelHelper.getRequest(objectModel);

        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        boolean memberOnly = zoneItem.getServiceParameters().getValue("memberOnly", false, false);

        String[] filterCategoriesArray = zoneItem.getServiceParameters().getValue("filterCategories", false, new String[0]);
        Set<String> filterCategories = Set.of(filterCategoriesArray);

        contentHandler.startDocument();
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("zoneItemId", zoneItem.getId());
        XMLUtils.startElement(contentHandler, "projects", attrs);

        if (_currentUserProvider.getUser() != null)
        {
            // Normalized, case insensitive comparator on project titles.
            Function<Project, String> getProjectTitle = Project::getTitle;
            Comparator<Project> projectTitleComparator = Comparator.comparing(getProjectTitle.andThen(StringUtils::stripAccents), String.CASE_INSENSITIVE_ORDER);

            Stream<Project> projectStream;
            if (memberOnly)
            {
                projectStream = _projectManager.getUserProjects(_currentUserProvider.getUser(), filterCategories)
                        .keySet()
                        .stream()
                        .sorted(projectTitleComparator);
            }
            else
            {
                projectStream = _projectManager.getProjects(filterCategories).stream()
                    .filter(project -> !project.getInscriptionStatus().equals(InscriptionStatus.PRIVATE) || _projectMemberManager.isProjectMember(project, _currentUserProvider.getUser()))
                    .sorted(projectTitleComparator);
            }

            // limit
            if (max > 0)
            {
                projectStream = projectStream.limit(max);
            }

            // SAX the projects
            projectStream.forEach(p -> _saxProject(p));
        }

        XMLUtils.endElement(contentHandler, "projects");
        contentHandler.endDocument();
    }

    private void _saxProject(Project project)
    {
        try
        {
            Site site = project.getSite();
            if (site == null)
            {
                // log and exit prematurely
                if (getLogger().isWarnEnabled())
                {
                    getLogger().warn(String.format("The project '%s' does not have any associated site.", project.getTitle()));
                }

                return;
            }

            UserIdentity userIdentity = _currentUserProvider.getUser();
            boolean hasAccess = _projectMemberManager.isProjectMember(project, userIdentity);

            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", project.getId());
            attrs.addCDATAAttribute("name", project.getName());
            attrs.addCDATAAttribute("siteName", site.getName());
            attrs.addCDATAAttribute("hasAccess", Boolean.toString(hasAccess));
            InscriptionStatus inscriptionStatus = project.getInscriptionStatus();
            attrs.addCDATAAttribute("inscriptionStatus", inscriptionStatus.toString());
            if (!hasAccess && !inscriptionStatus.equals(InscriptionStatus.PRIVATE) && userIdentity != null)
            {
                attrs.addCDATAAttribute("inPopulations", Boolean.toString(_projectManager.isUserInProjectPopulations(project, userIdentity)));
            }

            XMLUtils.startElement(contentHandler, "project", attrs);

            XMLUtils.createElement(contentHandler, "title", project.getTitle());
            String siteUrl = site.getUrl();
            if (siteUrl != null)
            {
                XMLUtils.createElement(contentHandler, "url", site.getUrl());
            }
            XMLUtils.createElement(contentHandler, "description", StringUtils.defaultString(project.getDescription()));

            site.illustrationToSAX(contentHandler);
            project.coverImageToSAX(contentHandler);

            // Managers
            XMLUtils.startElement(contentHandler, "managers");
            Arrays.stream(project.getManagers())
                    .map(_userManager::getUser)
                    .filter(Objects::nonNull)
                    .forEach(LambdaUtils.wrapConsumer(m -> _userHelper.saxUser(m, contentHandler, "manager")));
            XMLUtils.endElement(contentHandler, "managers");

            // Tags
            XMLUtils.startElement(contentHandler, "projectTags");
            project.getTags().stream()
                    .forEach(_saxDefaultTag(_projectTagProviderEP, "projectTag"));
            XMLUtils.endElement(contentHandler, "projectTags");

            // Keywords
            XMLUtils.startElement(contentHandler, "keywords");
            Arrays.stream(project.getKeywords())
                    .forEach(LambdaUtils.wrapConsumer(keyword -> XMLUtils.createElement(contentHandler, "keyword", keyword)));
            XMLUtils.endElement(contentHandler, "keywords");

            // Categories
            XMLUtils.startElement(contentHandler, "categories");
            project.getCategories().stream()
                    .forEach(_saxCategory("category"));
            XMLUtils.endElement(contentHandler, "categories");

            // Stats
            XMLUtils.startElement(contentHandler, "statistics");
            _saxProjectStatistic("nbProjectMember", _projectMemberManager.getMembersCount(project));
            _saxProjectStatistic("nbProjectDocuments", _workspaceExplorerResourceDAO.getDocumentsCount(project));
            _saxProjectStatistic("nbProjectTasks", _workspaceTaskDAO.getTasksCount(project));
            _saxProjectStatistic("nbProjectThreads", _workspaceThreadDAO.getThreadsCount(project));
            XMLUtils.endElement(contentHandler, "statistics");

            XMLUtils.endElement(contentHandler, "project");
        }
        catch (SAXException | IOException e)
        {
            throw new RuntimeException(String.format("Unable to SAX project with id '%s'", project.getId()), e);
        }
    }

    private Consumer<String> _saxCategory(String tagName)
    {
        Map<String, Object> contextualParameter = new HashMap<>();
        Function<String, Function<TagProvider<Category>, Category>> getTagFromExtension = categoryId -> extension -> extension.getTag(categoryId, contextualParameter);
        Consumer<Category> saxCategory = LambdaUtils.wrapConsumer(category ->
        {
            XMLUtils.startElement(contentHandler, tagName);
            category.getTitle().toSAX(contentHandler, "title");
            _saxCategoryColor(category);
            XMLUtils.endElement(contentHandler, tagName);
        });

        return categoryId -> _categoryProviderEP.getExtensionsIds().stream()
                .map(_categoryProviderEP::getExtension)
                .map(getTagFromExtension.apply(categoryId))
                .filter(Objects::nonNull)
                .findFirst()
                .ifPresent(saxCategory);
    }

    private Consumer<String> _saxDefaultTag(AbstractTagProviderExtensionPoint<DefaultTag> provider, String tagName)
    {
        Map<String, Object> contextualParameter = new HashMap<>();
        Function<String, Function<TagProvider<DefaultTag>, DefaultTag>> getTagFromExtension = keywordId -> extension -> extension.getTag(keywordId, contextualParameter);
        Consumer<DefaultTag> saxTag = LambdaUtils.wrapConsumer(tag -> tag.getTitle().toSAX(contentHandler, tagName));

        return keywordId -> provider.getExtensionsIds().stream()
                .map(provider::getExtension)
                .map(getTagFromExtension.apply(keywordId))
                .filter(Objects::nonNull)
                .findFirst()
                .ifPresent(saxTag);
    }

    private void _saxProjectStatistic(String elementName, Long count)
    {
        ThrowingConsumer<Long> saxElement = value -> XMLUtils.createElement(contentHandler, elementName, Long.toString(value));

        Optional.ofNullable(count)
                .ifPresent(LambdaUtils.wrapConsumer(saxElement));
    }

    private void _saxCategoryColor(Category category) throws SAXException
    {
        Map<String, String> colors = _categoryHelper.getCategoryColor(category);

        XMLUtils.startElement(contentHandler, "color");
        colors.entrySet().stream()
            .forEach(LambdaUtils.wrapConsumer(entry -> XMLUtils.createElement(contentHandler, entry.getKey(), entry.getValue())));
        XMLUtils.endElement(contentHandler, "color");
    }

}
