/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Condition to notify news publication
 *
 */
public class SendNewsPublicationNotificationCondition extends AbstractContentWorkflowComponent implements EnhancedCondition
{
    /** The user manager */
    protected ContentTypesHelper _cTypeHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _cTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    public boolean passesCondition(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        WorkflowAwareContent content = getContent(transientVars);
        
        // check if content is a project news
        if (!_cTypeHelper.isInstanceOf(content, WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID))
        {
            return false;
        }
        
        // check if the current user is the content's author
        UserIdentity user = getUser(transientVars);
        if (!content.getCreator().equals(user))
        {
            return false;
        }
        
        // check if the news was not already be notified
        boolean alreadyNotified = content.getInternalDataHolder().getValue(SetNotifiedFunction.NOTIFIED_PROPERTY_NAME, false);
        return !alreadyNotified;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_SEND_NEWS_PUBLICATION_NOTIFICATION_CONDITION_LABEL");
    }
}
