/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.accesscontroller;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessController.Permission.PermissionType;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.right.RightsException;
import org.ametys.core.right.RightsExtensionPoint;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.WorkspaceAccessController;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectManager.UnknownCatalogSiteException;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.rights.PageAccessController;

import com.google.common.collect.Iterables;

/**
 * Automatically grand access to CMS on site related to a project where the user is manager
 * AND has "Plugins_Workspaces_Rights_Project_BOAccess" right on general site context of catalog site.
 * Grants the right on general context and sitemap context with the default admin profile
 */
public class BackOfficeAccessController implements AccessController, Serviceable, Contextualizable, PluginAware
{
    private static final String __BO_ACCESS_RIGHT_ID = "Plugins_Workspaces_Rights_Project_BOAccess";

    private static final String __ALL_PAGE_FAKE_CONTEXT = "$ALL_PAGES$";
    
    /** The rights extension point */
    protected RightsExtensionPoint _rightsEP;
    
    private String _adminProfileIdentifier;
    private RightProfilesDAO _rightProfileDAO;
    private SiteManager _siteManager;
    private ProjectManager _projectManager;
    private RightManager _rightManager;

    private Context _context;
    private String _id;

    public void service(ServiceManager manager) throws ServiceException
    {
        _rightProfileDAO = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _rightsEP = (RightsExtensionPoint) manager.lookup(RightsExtensionPoint.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public boolean isSupported(Object object)
    {
        return object instanceof String && ((String) object).startsWith("/cms")
                || object instanceof SitemapElement;
    }
    
    private Project _getProject(Object object)
    {
        Site site = null;
        
        if (object instanceof SitemapElement se)
        {
            site = se.getSite();
        }
        else
        {
            Request request = ContextHelper.getRequest(_context);
            String siteName = WebHelper.getSiteName(request);
            if (StringUtils.isNotBlank(siteName))
            {
                site = _siteManager.getSite(siteName);
            }
        }
        
        if (site != null)
        {
            List<Project> projectsForSite = _projectManager.getProjectsForSite(site);
            Project project = Iterables.getFirst(projectsForSite, null);
            return project;
        }
        else
        {
            return null;
        }
    }
    
    private boolean _isRightInAdminProfile(String rightId)
    {
        if (rightId == null)
        {
            return true; // READER profile
        }
        
        return _getRightsInAdminProfile().contains(rightId);
    }
    
    private boolean _hasBOAccessRight(UserIdentity user)
    {
        try
        {
            return _rightManager.hasRight(user, __BO_ACCESS_RIGHT_ID, "/site/" + _projectManager.getCatalogSiteName()) == RightResult.RIGHT_ALLOW;
        }
        catch (UnknownCatalogSiteException e)
        {
            // Ignore the no catalog sitename exception
            return false;
        }
    }
    
    private synchronized List<String> _getRightsInAdminProfile()
    {
        if (_adminProfileIdentifier == null)
        {
            _adminProfileIdentifier = Config.getInstance().getValue("workspaces.profile.managerdefault");
        }
        
        return _rightProfileDAO.getRights(_adminProfileIdentifier);
    }

    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (_isRightInAdminProfile(rightId))
        {
            Project project = _getProject(object);
            if (project != null && _projectManager.isManager(project, user) && _hasBOAccessRight(user))
            {
                return AccessResult.USER_ALLOWED;
            }
        }
        
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return getPermission(user, userGroups, null, object);
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Project project = _getProject(object);
        if (project != null && _projectManager.isManager(project, user) && _hasBOAccessRight(user))
        {
            return _getRightsInAdminProfile().stream().collect(Collectors.toMap(rightId -> rightId, rightId -> AccessResult.USER_ALLOWED));
        }
        else
        {
            return Map.of();
        }
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        if (_isRightInAdminProfile(rightId))
        {
            Project project = _getProject(object);
            if (project != null)
            {
                return Stream.of(project.getManagers())
                        .distinct()
                        .filter(this::_hasBOAccessRight)
                        .collect(Collectors.toMap(user -> user, user -> AccessResult.USER_ALLOWED));
            }
        }
        
        return Map.of();
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return getPermissionByUser(null, object);
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Map.of();
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        if (_isRightInAdminProfile(rightId) && workspacesContexts.contains("/cms"))
        {
            Project project = _getProject("/cms");
            if (project != null && _projectManager.isManager(project, user) && _hasBOAccessRight(user))
            {
                return true; // Assume that admin profile is not empty
            }
        }

        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return hasUserAnyPermissionOnWorkspace(workspacesContexts, user, userGroups, null);
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult permission, Object object)
    {
        switch (permission)
        {
            case USER_ALLOWED:
            case UNKNOWN:
                
                return new AccessExplanation(
                        getId(),
                        permission,
                        new I18nizableText(
                                "plugin.workspaces",
                                "PLUGINS_WORKSPACES_BACK_OFFICE_ACCESS_CONTROLLER_" + permission.name() + "_EXPLANATION",
                                Map.of("right", _rightsEP.getExtension(__BO_ACCESS_RIGHT_ID).getLabel(), "objectLabel", _getObjectLabelForExplanation(object))
                            )
                        );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), permission);
        }
    }

    private I18nizableText _getObjectLabelForExplanation(Object object)
    {
        if (object instanceof String str)
        {
            if (__ALL_PAGE_FAKE_CONTEXT.equals(str))
            {
                return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_BACK_OFFICE_ACCESS_CONTROLLER_ALL_PAGE_EXPLANATION_OBJECT_LABEL");
            }
            else
            {
                return new I18nizableText("plugin.web", "PLUGINS_WEB_WORKSPACE_ACCESS_CONTROLLER_CONTEXT_LABEL", Map.of("object", new I18nizableText(str)));
            }
        }
        else if (object instanceof Page)
        {
            return new I18nizableText("plugin.web", "PLUGINS_WEB_PAGE_ACCESS_CONTROLLER_PAGE_CONTEXT_EXPLANATION_LABEL", Map.of("title", getObjectLabel(object)));
        }
        else if (object instanceof Sitemap sitemap)
        {
            return new I18nizableText("plugin.web", "PLUGINS_WEB_PAGE_ACCESS_CONTROLLER_SITEMAP_CONTEXT_EXPLANATION_LABEL", Map.of("name", new I18nizableText(sitemap.getSitemapName().toUpperCase())));
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof String str)
        {
            if (__ALL_PAGE_FAKE_CONTEXT.equals(str))
            {
                return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_BACK_OFFICE_ACCESS_CONTROLLER_ALL_PAGE_OBJECT_LABEL");
            }
            else
            {
                return WorkspaceAccessController.GENERAL_CONTEXT_CATEGORY;
            }
        }
        else if (object instanceof Page page)
        {
            return new I18nizableText(PageAccessController.getPageObjectLabel(page));
        }
        else if (object instanceof Sitemap sitemap)
        {
            return new I18nizableText("plugin.web", "PLUGINS_WEB_PAGE_ACCESS_CONTROLLER_SITEMAP_CONTEXT_LABEL", Map.of("name", new I18nizableText(sitemap.getSitemapName().toUpperCase())));
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups)
    {
        Map<ExplanationObject, Map<Permission, AccessExplanation>> result = new HashMap<>();
        
        AccessResult accessResult = getPermission(identity, groups, null, null);
        if (accessResult != AccessResult.UNKNOWN)
        {
            AccessExplanation explanation = getStandardAccessExplanation(accessResult, "/cms");
            
            Map<Permission, AccessExplanation> contextPermission = Map.of(
                    new Permission(PermissionType.PROFILE, _adminProfileIdentifier), explanation,
                    new Permission(PermissionType.READ, null), explanation
                    );
            
            result.put(
                    getExplanationObject("/cms"),
                    contextPermission
            );
            
            explanation = getStandardAccessExplanation(accessResult, __ALL_PAGE_FAKE_CONTEXT);
            
            contextPermission = Map.of(
                    new Permission(PermissionType.PROFILE, _adminProfileIdentifier), explanation,
                    new Permission(PermissionType.READ, null), explanation
                    );
            
            result.put(
                    getExplanationObject(__ALL_PAGE_FAKE_CONTEXT),
                    contextPermission
            );
        }
        
        return result;
    }

    public I18nizableText getObjectCategory(Object object)
    {
        if (object instanceof String)
        {
            if (__ALL_PAGE_FAKE_CONTEXT.equals(object))
            {
                return PageAccessController.PAGE_CONTEXT_CATEGORY;
            }
            
            return WorkspaceAccessController.GENERAL_CONTEXT_CATEGORY;
        }
        else if (object instanceof SitemapElement)
        {
            return PageAccessController.PAGE_CONTEXT_CATEGORY;
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }
    
    public int getObjectPriority(Object object)
    {
        if (__ALL_PAGE_FAKE_CONTEXT.equals(object))
        {
            return 10;
        }
        return AccessController.super.getObjectPriority(object);
    }
    
    public String getId()
    {
        return _id;
    }
}
