/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.report;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.tag.Tag;
import org.ametys.core.group.GroupManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.activities.ActivityHelper;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.keywords.KeywordsDAO;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.objects.Project.InscriptionStatus;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Report projects with theirs managers/users
 */
public class ReportGenerator extends ServiceableGenerator implements Contextualizable
{
    /** Time format for csv export */
    protected static final String __DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm";
    
    /** Zoned dateTime formatter */
    protected static final DateTimeFormatter __ZONED_DATE_FORMATER = DateTimeFormatter.ofPattern(__DATE_TIME_FORMAT);

    /** DateTime formatter */
    protected static final SimpleDateFormat __DATE_FORMATER = new SimpleDateFormat(__DATE_TIME_FORMAT);
    
    
    /** number of columns for a project */
    protected static final int __REPORT_COLUMNS_SIZE_PROJECT = 8;
    
    /** number of columns for a member */
    protected static final int __REPORT_COLUMNS_SIZE_MEMBER = 8;
    
    /** Report Project Manager */
    protected ReportHelper _reportProjectManager;
    
    /** The project manager */
    protected ProjectManager _projectManager;
    
    /** Site manager */
    protected SiteManager _siteManager;

    /** Repository */
    protected Repository _repository;

    /** User Manager */
    protected UserManager _userManager;
    
    /** Group Manager */
    protected GroupManager _groupManager;
    
    /** Project Member Manager */
    protected ProjectMemberManager _projectMemberManager;
    
    /** User Directory Helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    /** The keywords DAO */
    protected KeywordsDAO _keywordsDAO;
    
    /** Avalon context */
    protected Context _context;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _reportProjectManager = (ReportHelper) smanager.lookup(ReportHelper.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _repository = (Repository) smanager.lookup(AbstractRepository.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _groupManager = (GroupManager) smanager.lookup(GroupManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) smanager.lookup(ProjectMemberManager.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _keywordsDAO = (KeywordsDAO) manager.lookup(KeywordsDAO.ROLE);
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ContextHelper.getRequest(_context);
        
        String lang = parameters.getParameter("lang", null);
        String[] projectsValues = request.getParameterValues("project");
        String[] categoriesValues = request.getParameterValues("category");

        // Filters because the js may send the parameter empty (which will create a non-emply list with an empty string)
        List<String> projects = projectsValues == null ? Collections.EMPTY_LIST : Arrays.asList(projectsValues)
                .stream()
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toList());
        
        List<String> categories = categoriesValues == null ? Collections.EMPTY_LIST : Arrays.asList(categoriesValues)
                .stream()
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toList());

        boolean reportWithMembers = "true".equals(request.getParameter("with-members"));
        boolean reportWithManagers = "true".equals(request.getParameter("with-managers"));
        
        contentHandler.startDocument();
        List<Project> availableProjects = _reportProjectManager.getAvailableProjects(projects, categories);
        if (availableProjects != null && !availableProjects.isEmpty())
        {
            XMLUtils.createElement(contentHandler, "text", generateProjects(availableProjects, reportWithMembers, reportWithManagers, lang));
        }
        else
        {
            XMLUtils.createElement(contentHandler, "text");
        }
        contentHandler.endDocument();
    }

    /**
     * Generate the list of projects
     * @param projects list of projects to generate
     * @param reportWithMembers also report members
     * @param reportWithManagers also report managers
     * @param lang language used to parse agents (if not available, another langage will still be used)
     * @return A String that will represent the csv
     */
    protected String generateProjects(List<Project> projects, boolean reportWithMembers, boolean reportWithManagers, String lang)
    {
        // Languages used to use one of them on each content to report
        List<String> langages = getCatalogSiteLangages();
        
        StringBuilder sb = new StringBuilder();
        sb.append("\ufeff"); // BOM character
        sb.append("\"" + translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_TYPE", lang) + "\";");
        sb.append(getProjectCsvHeader(reportWithMembers, reportWithManagers, lang));
        
        sb.append("\n");
        
        // categories
        sb.append(projects.stream()
                .map(project -> translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT", lang) + ";" + projectToCsv(project, reportWithMembers, reportWithManagers, lang, langages))
                .collect(Collectors.joining("\n")));

        return sb.toString();
    }
    
    /**
     * Get the languages of the catalog site
     * @return the list of catalog site languages
     */
    protected List<String> getCatalogSiteLangages()
    {
        String catalogSiteName = _projectManager.getCatalogSiteName();
        if (StringUtils.isNotEmpty(catalogSiteName))
        {
            Site siteCatalog = _siteManager.getSite(catalogSiteName);
            if (siteCatalog != null)
            {
                return siteCatalog.getSitemaps().stream()
                    .map(Sitemap::getSitemapName)
                    .collect(Collectors.toList());
            }
        }
        return null;
    }
    
    /**
     * Format a String to csv
     * @param line strings to transform
     * @return a string representing the input strings
     */
    protected static String formatToCsv(String... line)
    {
        return Arrays.stream(line)
                .map(org.ametys.core.util.StringUtils::sanitizeCsv)
                .collect(Collectors.joining(";"));
    }
    
    /**
     * Generate the header for projects
     * @param reportWithMembers also report members
     * @param reportWithManagers also report managers
     * @param lang language to use to create header
     * @return a String representing the csv header
     */
    protected String getProjectCsvHeader(boolean reportWithMembers, boolean reportWithManagers, String lang)
    {
        StringBuilder sb = new StringBuilder(formatToCsv(
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_NAME", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_DESC", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_CATEGORY", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_CREATION_DATE", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_ACTIVITY_DATE", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_KEYWONDS", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_MANAGERS", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_VISIBILITY", lang)));
        
        if (reportWithMembers || reportWithManagers)
        {
            sb.append(";");
            sb.append(getMemberCsvHeader(lang));
        }
        
        return sb.toString();
    }
    
    private String getMemberCsvHeader(String lang)
    {
        return formatToCsv(
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_TITLE", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_FIRSTNAME", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_LASTNAME", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_FUNCTION", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_ORGANISATION", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_PHONE", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_MAIL", lang),
                translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_MEMBER_SKILLS", lang));
    }
    
    private String projectToCsv(Project project, boolean reportWithMembers, boolean reportWithManagers, String lang, List<String> langages)
    {
        Set<String> projectCategories = project.getCategories();
        String mainCategory = projectCategories.size() > 0 ? projectCategories.iterator().next() : "";
                
        String lastActivityDate = getProjectLastActivityDate(project);
        
        String projectmanager = Arrays.stream(project.getManagers())
            .map(_userManager::getUser)
            .filter(Objects::nonNull)
            .map(user -> user.getFullName() + " (" + user.getIdentity().getPopulationId() + ")")
            .collect(Collectors.joining(","));
        
        InscriptionStatus inscriptionStatus = project.getInscriptionStatus();
        String inscriptionStatusString = InscriptionStatus.PRIVATE.equals(inscriptionStatus) ? translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_INSCRIPTION_STATUS_PRIVATE", lang)
                : InscriptionStatus.MODERATED.equals(inscriptionStatus) ? translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_INSCRIPTION_STATUS_MODERATED", lang)
                        : InscriptionStatus.OPEN.equals(inscriptionStatus) ? translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_INSCRIPTION_STATUS_OPEN", lang) : "";
        
        
        StringBuilder sb = new StringBuilder(formatToCsv(
                project.getTitle(),
                StringUtils.defaultString(project.getDescription(), "").replaceAll("[\r\n]+", " / "),
                mainCategory,
                __ZONED_DATE_FORMATER.format(project.getCreationDate()),
                lastActivityDate,
                getKeywords(project, lang),
                projectmanager,
                inscriptionStatusString
            ));
        
        int projectPadding = (reportWithMembers || reportWithManagers) ? __REPORT_COLUMNS_SIZE_MEMBER : 0;
        
        sb.append(StringUtils.repeat(";-", projectPadding));
        
        if (reportWithManagers)
        {
            // project managers
            String managerPrefix = "\"" + translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_MANAGER", lang) + "\";\"" + project.getTitle() + "\"" + StringUtils.repeat(";-", __REPORT_COLUMNS_SIZE_PROJECT - 1) + ";";
                    
            String managersCsv = Arrays.stream(project.getManagers())
                .map(userIdentity -> createProjectMemberFromUser(userIdentity, lang,  langages))
                .filter(Objects::nonNull)
                .map(ProjectMember::toCsv)
                .map(managerCsv -> managerPrefix + managerCsv)
                .collect(Collectors.joining("\n"));
            if (StringUtils.isNotEmpty(managersCsv))
            {
                sb.append("\n");
                sb.append(managersCsv);
            }
        }
        
        if (reportWithMembers)
        {
            // project members
            String memberPrefix = "\"" + translateKey("PLUGIN_WORKSPACES_SERVICE_REPORTS_CSV_HEADER_PROJECT_MEMBER", lang) + "\";\"" + project.getTitle() + "\"" + StringUtils.repeat(";-", __REPORT_COLUMNS_SIZE_PROJECT - 1) + ";";
            
            String projectMembersCsv = getProjectMembersCsv(project, lang, langages).stream()
                    .map(memberCsv -> memberPrefix + memberCsv)
                    .collect(Collectors.joining("\n"));
            
            if (StringUtils.isNotEmpty(projectMembersCsv))
            {
                sb.append("\n");
                sb.append(projectMembersCsv);
            }
        }
        

        
        return sb.toString();
    }
    
    /**
     * Get the project's keywords
     * @param project the project
     * @param lang the current language
     * @return the keywords as label
     */
    protected String getKeywords(Project project, String lang)
    {
        List<String> keywordLabels = new ArrayList<>();
        
        for (String keyword : project.getKeywords())
        {
            Tag tag = _keywordsDAO.getTag(keyword, Collections.emptyMap());
            if (tag != null)
            {
                String title = _i18nUtils.translate(tag.getTitle(), lang);
                keywordLabels.add(title);
            }
        }
        
        return StringUtils.join(keywordLabels, ",");
    }
    
    /**
     * Get the last activity date of the project
     * @param project project to analyze
     * @return a formatted date
     */
    protected String getProjectLastActivityDate(Project project)
    {
        StringExpression projectExpression = new StringExpression(AbstractWorkspacesActivityType.PROJECT_NAME, Operator.EQ, project.getName());
        String xpathQuery = ActivityHelper.getActivityXPathQuery(projectExpression);
        
        Session session = null;
        try
        {
            session  = _repository.login();
            @SuppressWarnings("deprecation")
            Query query = session.getWorkspace().getQueryManager().createQuery(xpathQuery, Query.XPATH);
            NodeIterator nodes = query.execute().getNodes();
            if (nodes.hasNext())
            {
                Node node = (Node) nodes.next();
                Calendar date = node.getProperty("ametys:date").getValue().getDate();
                return __DATE_FORMATER.format(date.getTime());
            }
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("An error occured executing the JCR query : " + xpathQuery, ex);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
        
        return "";
    }
    
    /**
     * Get a {@link ProjectMember} from a user
     * @param userIdentity user
     * @param lang language to use to parse user
     * @param projectLanguages languages of the catalog site
     * @return the user as a {@link ProjectMember}
     */
    protected ProjectMember createProjectMemberFromUser(UserIdentity userIdentity, String lang, List<String> projectLanguages)
    {
        User user = _userManager.getUser(userIdentity);
        if (user == null)
        {
            return null;
        }
        Content userContent = null;
        if (StringUtils.isNotEmpty(lang))
        {
            userContent = _userDirectoryHelper.getUserContent(userIdentity, lang);
        }
        
        // If requested language is not available, find one that is available
        if (userContent == null)
        {
            userContent = projectLanguages != null ? projectLanguages.stream()
                .map(l -> _userDirectoryHelper.getUserContent(userIdentity, l))
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(null) : null;
        }
        
        if (userContent == null)
        {
            return new ProjectMember(UserIdentity.userIdentityToString(userIdentity),
                    "", 
                    user.getFirstName(), 
                    user.getLastName(), 
                    "",
                    "", 
                    "", 
                    StringUtils.defaultIfEmpty(user.getEmail(), ""), 
                    "");
        }
        
        ModelAwareDataHolder dataHolder = userContent.getDataHolder();
        ContentValue[] skillsContents = dataHolder.getValue("skills");
        String skills = skillsContents != null ? Arrays.stream(skillsContents)
                .filter(Objects::nonNull)
                .map(ContentValue::getContent)
                .filter(Objects::nonNull)
                .map(skill -> (String) skill.getDataHolder().getValue("title"))
                .collect(Collectors.joining(","))
                : "";
                
        
        return new ProjectMember(UserIdentity.userIdentityToString(userIdentity),
                StringUtils.defaultIfEmpty(dataHolder.getValue("title"), ""), 
                user.getFirstName(), 
                user.getLastName(), 
                StringUtils.defaultIfEmpty(dataHolder.getValue("function"), ""),
                StringUtils.defaultIfEmpty(dataHolder.getValue("organisation"), ""), 
                StringUtils.defaultIfEmpty(dataHolder.getValue("phone"), ""), 
                StringUtils.defaultIfEmpty(user.getEmail(), ""), 
                skills);
    }
    
    /**
     * Get the csv for members
     * @param project project to use
     * @param lang language to use to parse user
     * @param languages languages of the catalog site
     * @return users as csv
     */
    protected List<String> getProjectMembersCsv(Project project, String lang, List<String> languages)
    {
        
        
        return _projectMemberManager.getProjectMembers(project, true)
                .stream()
                .map(member -> createProjectMemberFromUser(member.getUser().getIdentity(), lang, languages))
                .sorted(Comparator.comparing(ProjectMember::getSortableName))
                .map(ProjectMember::toCsv)
                .collect(Collectors.toList());
    }
        
    /**
     * Small shortcut to translate a key
     * @param key i18n key
     * @param lang requested language
     * @return the translated key
     */
    protected String translateKey(String key, String lang)
    {
        I18nizableText i18n = new I18nizableText("plugin.workspaces", key);
        return _i18nUtils.translate(i18n, lang);
    }
    
    static class ProjectMember
    {
        private String _id;
        private String _title;
        private String _firstname;
        private String _lastname;
        private String _role;
        private String _structure;
        private String _telephone;
        private String _email;
        private String _skills;
        
        ProjectMember(String id, String title, String firstname, String lastname, String role, String structure, String telephone, String email, String skills)
        {
            _id = id;
            _title = title;
            _firstname = firstname;
            _lastname = lastname;
            _role = role;
            _structure = structure;
            _telephone = telephone;
            _email = email;
            _skills = skills;
        }
        
        public String getId()
        {
            return _id;
        }
        
        public String getSortableName()
        {
            return _lastname + " " + _firstname;
        }
        
        @Override
        public boolean equals(Object obj)
        {
            return _id != null && obj instanceof ProjectMember && _id.equals(((ProjectMember) obj).getId());
        }
        
        /**
         * Format a {@link ProjectMember} to a csv line
         * @return a String representing a {@link ProjectMember} in a csv format
         */
        public String toCsv()
        {
            return formatToCsv(_title,
                _firstname,
                _lastname,
                _role,
                _structure,
                _telephone,
                _email,
                _skills);
        }

        @Override
        public int hashCode()
        {
            return _id.hashCode();
        }
    }
}
