/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.actions;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.tasks.Task;

/**
 * Action providing events for the calendars of a project.
 * The list of calendar can also be filtered.
 */
public class GetTaskForICSAction extends ServiceableAction
{
    private static final String __FULL_DAY_PATTERN = "uuuuMMdd";
    
    /** The object resolver */
    protected AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);
        String id = request.getParameter("id");
        Task task = _resolver.resolveById(id);
        
        if (task != null && task.getDueDate() != null)
        {
            Map<String, Object> taskMap = new HashMap<>();
            taskMap.put("id", task.getId());
            taskMap.put("title", task.getLabel());
            taskMap.put("description", task.getDescription());
            taskMap.put("fullDay", true);
            taskMap.put("keywords", task.getTags());
            LocalDate startDateEvent = task.getDueDate();
            taskMap.put("startDate", startDateEvent.format(DateTimeFormatter.BASIC_ISO_DATE));
            taskMap.put("endDateNextDay", startDateEvent.plusDays(1).format(DateTimeFormatter.BASIC_ISO_DATE));
            taskMap.put("creationDate", DateUtils.zonedDateTimeToString(task.getCreationDate(), task.getCreationDate().getZone(), __FULL_DAY_PATTERN));
            taskMap.put("lastModified", DateUtils.zonedDateTimeToString(task.getLastModified(), task.getLastModified().getZone(), __FULL_DAY_PATTERN));
            taskMap.put("tags", task.getTags());
            taskMap.put("recurrenceType", "NEVER");
            result.put("events", taskMap);
        }
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }
}
