/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.jcr;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObjectHelper;
import org.ametys.plugins.workspaces.tasks.Task;

/**
 * JCR implementation of a task
 */
public class JCRTask extends DefaultAmetysObject<JCRTaskFactory> implements Task
{
    /** Attribute Taskid */
    public static final String ATTRIBUTE_TASK_ID = "id";
    /** Attribute Taskid */
    public static final String ATTRIBUTE_TASKSLISTID = "tasksListId";
    /** Attribute Taskid */
    public static final String ATTRIBUTE_TASKSLISTPOSITION = "tasksListPosition";
    /** Attribute Label */
    public static final String ATTRIBUTE_LABEL = "label";
    /** Attribute Description */
    public static final String ATTRIBUTE_DESCRIPTION = "description";
    /** Attribute Startdate */
    public static final String ATTRIBUTE_STARTDATE = "startDate";
    /** Attribute DueDate */
    public static final String ATTRIBUTE_DUEDATE = "dueDate";
    /** Attribute IsClosed */
    public static final String ATTRIBUTE_ISCLOSED = "isClosed";
    /** Attribute CloseDate */
    public static final String ATTRIBUTE_CLOSEDATE = "closeDate";
    /** Attribute CloseAuthor */
    public static final String ATTRIBUTE_CLOSEAUTHOR = "closeAuthor";
    /** Attribute Author */
    public static final String ATTRIBUTE_AUTHOR = "author";
    /** Attribute Assignments */
    public static final String ATTRIBUTE_ASSIGNMENTS = "assignments";
    /** Attribute Creationdate */
    public static final String ATTRIBUTE_CREATIONDATE = "creationDate";
    /** Attribute Lastmodified */
    public static final String ATTRIBUTE_LASTMODIFIED = "lastModified";
    /** Attribute Attachments */
    public static final String ATTRIBUTE_ATTACHMENTS = "attachments";
    /** Attribute CheckList */
    public static final String ATTRIBUTE_CHECKLIST = "checkList";
    /** Attribute Label for checkList */
    public static final String ATTRIBUTE_CHECKLIST_LABEL = "label";
    /** Attribute IsChecked for checkList */
    public static final String ATTRIBUTE_CHECKLIST_ISCHECKED = "isChecked";
    /** Attribute Tags */
    public static final String ATTRIBUTE_TAGS = "tags";
    
    /**
     * Default constructor for the JCRTask
     * @param node The task node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public JCRTask(Node node, String parentPath, JCRTaskFactory factory)
    {
        super(node, parentPath, factory);
    }

    public String getTaskListId()
    {
        return getValue(ATTRIBUTE_TASKSLISTID);
    }
    
    public void setTasksListId(String taskListId)
    {
        setValue(ATTRIBUTE_TASKSLISTID, taskListId);
    }
    
    public Long getPosition()
    {
        return getValue(ATTRIBUTE_TASKSLISTPOSITION);
    }
    
    public void setPosition(Long position)
    {
        setValue(ATTRIBUTE_TASKSLISTPOSITION, position);
    }

    public String getLabel()
    {
        return getValue(ATTRIBUTE_LABEL);
    }

    public void setLabel(String label)
    {
        setValue(ATTRIBUTE_LABEL, label);
    }
    
    public String getDescription()
    {
        return getValue(ATTRIBUTE_DESCRIPTION);
    }

    public void setDescription(String description)
    {
        setValue(ATTRIBUTE_DESCRIPTION, description);
    }
    
    public LocalDate getStartDate()
    {
        return getValue(ATTRIBUTE_STARTDATE);
    }
    
    public void setStartDate(LocalDate startDate)
    {
        setValue(ATTRIBUTE_STARTDATE, startDate);
    }

    public LocalDate getDueDate()
    {
        return getValue(ATTRIBUTE_DUEDATE);
    }

    public void setDueDate(LocalDate dueDate)
    {
        setValue(ATTRIBUTE_DUEDATE, dueDate);
    }
    
    public boolean isClosed()
    {
        return getValue(ATTRIBUTE_ISCLOSED, false, false);
    }
    
    public void close(boolean isClosed)
    {
        setValue(ATTRIBUTE_ISCLOSED, isClosed);
    }
    
    public UserIdentity getCloseAuthor()
    {
        return getValue(ATTRIBUTE_CLOSEAUTHOR);
    }
    
    public void setCloseAuthor(UserIdentity author)
    {
        setValue(ATTRIBUTE_CLOSEAUTHOR, author);
    }

    public LocalDate getCloseDate()
    {
        return getValue(ATTRIBUTE_CLOSEDATE);
    }
    
    public void setCloseDate(LocalDate closedDate)
    {
        setValue(ATTRIBUTE_CLOSEDATE, closedDate);
    }
    
    public UserIdentity getAuthor()
    {
        return getValue(ATTRIBUTE_AUTHOR);
    }
    
    public void setAuthor(UserIdentity author)
    {
        setValue(ATTRIBUTE_AUTHOR, author);
    }
    
    public List<UserIdentity> getAssignments()
    {
        UserIdentity[] users = getValue(ATTRIBUTE_ASSIGNMENTS, false, new UserIdentity[0]);
        return Arrays.asList(users);
    }
    
    public void setAssignments(List<UserIdentity> assignments)
    {
        setValue(ATTRIBUTE_ASSIGNMENTS, assignments.toArray(new UserIdentity[assignments.size()]));
    }

    public ZonedDateTime getCreationDate()
    {
        return getValue(ATTRIBUTE_CREATIONDATE);
    }
    
    public void setCreationDate(ZonedDateTime creationDate)
    {
        setValue(ATTRIBUTE_CREATIONDATE, creationDate);
    }
    
    public ZonedDateTime getLastModified()
    {
        return getValue(ATTRIBUTE_LASTMODIFIED);
    }
    
    public void setLastModified(ZonedDateTime lastModifiedDate)
    {
        setValue(ATTRIBUTE_LASTMODIFIED, lastModifiedDate);
    }
    
    public List<Binary> getAttachments()
    {
        Binary[] attachements = getValue(ATTRIBUTE_ATTACHMENTS, false, new Binary[0]);
        return Arrays.asList(attachements);
    }
    
    public void setAttachments(List<Binary> attachments)
    {
        setValue(ATTRIBUTE_ATTACHMENTS, attachments.toArray(new Binary[attachments.size()]));
    }

    public void tag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.tag(this, tag);
    }

    public void untag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.untag(this, tag);
    }

    public Set<String> getTags() throws AmetysRepositoryException
    {
        return TaggableAmetysObjectHelper.getTags(this);
    }

    public List<CheckItem> getCheckList()
    {
        ModifiableModelAwareRepeater repeater = getRepeater(ATTRIBUTE_CHECKLIST, true);
        return repeater.getEntries()
                    .stream()
                    .map(e -> new CheckItem(e.getValue(ATTRIBUTE_CHECKLIST_LABEL), e.getValue(ATTRIBUTE_CHECKLIST_ISCHECKED, false, false)))
                    .collect(Collectors.toList());
    }

    public void setCheckListItem(List<CheckItem> checkItems)
    {
        ModifiableModelAwareRepeater repeater = getRepeater(ATTRIBUTE_CHECKLIST, true);
        for (ModifiableModelAwareRepeaterEntry entry : repeater.getEntries())
        {
            repeater.removeEntry(entry.getPosition());
        }
        
        for (CheckItem checkItem : checkItems)
        {
            ModifiableModelAwareRepeaterEntry entry = repeater.addEntry();
            
            entry.setValue(ATTRIBUTE_CHECKLIST_LABEL, checkItem.getLabel());
            entry.setValue(ATTRIBUTE_CHECKLIST_ISCHECKED, checkItem.isChecked());
        }
    }
    
    private ModifiableModelLessDataHolder _getCommentsDataHolder()
    {
        try
        {
            Node baseNode = getBaseNode();
            if (!baseNode.hasNode("ametys:comments"))
            {
                baseNode.addNode("ametys:comments", "ametys:compositeMetadata");
            }
            Node commentsNode = baseNode.getNode("ametys:comments");
            
            // FIXME WORKSPACES-1614: there is a double level of comments nodes
            if (!commentsNode.hasNode("ametys:comments"))
            {
                commentsNode.addNode("ametys:comments", "ametys:compositeMetadata");
            }
            Node secondLevelCommentsNode = commentsNode.getNode("ametys:comments");
            
            baseNode.getSession().save();
            
            ModifiableRepositoryData repositoryData = new JCRRepositoryData(secondLevelCommentsNode);
            return new DefaultModifiableModelLessDataHolder(_getFactory().getUnversionedDataTypeExtensionPoint(), repositoryData);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public Comment createComment()
    {
        
        return new Comment(_getCommentsDataHolder());
    }
    
    public Comment createComment(String commentId, ZonedDateTime creationDate)
    {
        return new Comment(_getCommentsDataHolder(), commentId, creationDate);
    }
    
    public Comment getComment(String commentId) throws AmetysRepositoryException
    {
        return Comment.getComment(_getCommentsDataHolder(), commentId);
    }
    
    public List<Comment> getComments(boolean includeNotValidatedComments, boolean includeValidatedComments) throws AmetysRepositoryException
    {
        return Comment.getComments(_getCommentsDataHolder(), includeNotValidatedComments, includeValidatedComments);
    }

    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getTaskModel());
    }
}
