/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.user;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.User;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.web.repository.page.ZoneItem;

import com.google.common.collect.Sets;

/**
 * Component to manager user
 */
public class UserComponent implements Component, Serviceable
{
    /** The avalon role */
    public static final String ROLE = UserComponent.class.getName();
    
    private UserManager _userManager;
    private UserPopulationDAO _userPopulationDAO;
    private UserHelper _userHelper;
    private AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _userPopulationDAO = (UserPopulationDAO) manager.lookup(UserPopulationDAO.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Search users in the given site
     * @param siteName The site name
     * @param count Number of results to return
     * @param offset Offset to start results
     * @param search Text to filter search
     * @param zoneItemId the zoneItemId of the catalog service, used to get allowed populations
     * @return The users as JSON
     */
    @Callable
    public Map<String, Object> getUsers(String siteName, int count, int offset, String search, String zoneItemId)
    {
        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
        ModelAwareDataHolder serviceDataHolder = zoneItem.getServiceParameters();
        String[] populationIds = serviceDataHolder.getValue("populationIds") != null ? serviceDataHolder.getValue("populationIds") : new String[0];

        Map<String, Object> parameters = Collections.singletonMap("pattern", search);
        
        List<User> usersByContext = new ArrayList<>();
        
        if (populationIds.length == 0)
        {
            Set<String> contexts = Sets.newHashSet("/sites/" + siteName, "/sites-fo/" + siteName);
            
            usersByContext = _userManager.getUsersByContext(contexts, count, offset, parameters, true, true);
        }
        else
        {
            List<UserPopulation> userPopulations = Arrays.stream(populationIds)
                    .map(userPopulationId -> _userPopulationDAO.getUserPopulation(userPopulationId))
                    .collect(Collectors.toList());
            usersByContext = _userManager.getUsers(userPopulations, count, offset, parameters, true);
        }
        
        List<Map<String, Object>> usersAsJson = usersByContext.stream()
            .map(_userHelper::user2json)
            .collect(Collectors.toList());
        
        return Collections.singletonMap("users", usersAsJson);
    }
}
