/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

Ametys = window.Ametys || {}
Ametys.form = Ametys.form || {}
Ametys.form.widget = Ametys.form.widget || {}
Ametys.form.widget.RichText = Ametys.form.widget.RichText || {}

Ametys.form.widget.RichText.RichTextConfiguration = {
    /**
     * @property {Object} _useCSSFile A map of category / css files to load. "" is the default category
     * @property {String[]} _useCSSFile.key List of css files to load in the inline editor for a category
     * @private
     */
    _useCSSFile: {},
    /**
     * @property {Object} _tags A map of category / handled tags. "" is the default category.
     * @property {Object} _tags.key Handled tags for a category
     * @private
     */
    _tags: {},
    /**
     * @property {Object} __styles A map of category / handled styles. "" is the default category.
     * @property {Object} __styles.key Handled styles for a category
     * @private
     */
    _styles: {},
    
    /**
     * @private
     * @property {Boolean} _initialized Is the RichTextConfiguration ready to be used? 1 means yes, and a negative file means files are beeing loaded
     */
    _initialized: false,
    
    /**
     * Validates the value among the existing validators for the inline editor
     * @param {String} value The current field value
     * @param {String} [category=""] The category to validates
     * @return {Boolean/String} validator.return
     *
     * - True if the value is valid
     * - An error message if the value is invalid
     */
    validates: function(value, category)
    {
        return RichTextConfigurationBase.validates(value, category);
    },
        
    /**
     * Converts the value on get content
     * @param {Ametys.form.field.RichText} field The richtext to convert
     * @param {tinymce.Editor} editor The current richtext editor
     * @param {Object} object The object of value to be modified
     * @param {String} [category=""] The category to apply.
     */
    convertOnGetContent: function(field, editor, object, category)
    {
        return RichTextConfigurationBase.convertOnGetContent(field, editor, object, category);
    },
    
    /**
     * Converts the value on set content
     * @param {Ametys.form.field.RichText} field The richtext to convert
     * @param {tinymce.Editor} editor The current richtext editor
     * @param {Object} object The object of value to be modified
     * @param {String} [category=""] The category to apply.
     */
    convertOnSetContent: function(field, editor, object, category)
    {
        return RichTextConfigurationBase.convertOnSetContent(field, editor, object, category);
    },
        
    /**
     * Get all added css files as one string
     * @param {String} [category=""] The category to apply.
     * @return {String} The comma-separated list of added files
     */
    getCSSFiles: function(category)
    {
        return RichTextConfigurationBase.getCSSFiles(category);
//        return RichTextConfigurationBase.getCSSFiles(category).replaceAll("/plugins/workspaces/resources", AmetysFront.getPluginResourcesPrefix("workspaces") );
    },
    
    /**
     * Get all supported tags as a tinymce string. See valid_element tinymce configuration doc for the exact format.
     * @param {String} [category=""] The category to apply.
     * @return {String} The supported tags.
     */
    getTags: function(category)
    {
        return RichTextConfigurationBase.getTags(category);
    },
    
    /**
     * Get all supported styles as a tinymce conf object. See valid_styles tinymce configuration doc for the exact format.
     * @param {String} [category=""] The category to apply.
     * @return {Object} The supported properties for the style attribute.
     */
    getStylesForTags: function(category)
    {
        return RichTextConfigurationBase.getStylesForTags(category);
    },
    
    /**
     * Get all supported classes as a tinymce conf object. See valid_classes tinymce configuration doc for the exact format.
     * @param {String} [category=""] The category to apply.
     * @return {Object} The supported properties for the style attribute.
     */
    getClassesForTags: function(category)
    {
        return RichTextConfigurationBase.getClassesForTags(category);
    },    
        
    /**
     * Get tag informations. 
     * @param {String} tagName The name of the tag to handle in the configuration
     * @param {String} [category=""] The category of configuration where to handle
     * @return {Object} See #handleTag to know more
     */
    getTag: function(tagName, category)
    {
        return RichTextConfigurationBase.getTag(tagName, category);
    },
        
    /**
     * Get styles informations of a type. 
     * @param {String} type The type of element to style. Such as "paragraph", "table", "link", "image", "ol", "ul"...
     * @param {String} [category=""] The category of configuration where to handle
     * @return {Object[]} The styles available by group. See #handleStyledGroup return value. Can be null or empty.
     */
    getStyledElements: function(type, category)
    {
        return RichTextConfigurationBase.getStyledElements(type, category);
    },
    
    /**
     * Handle validators
     * @param {Object} validators The validators to handle.
     */
    handleValidators: function(validators, category)
    {
        // Nothing to do, no validators on workspace richtext
    },
    
    /**
     * Handle convertors
     * @param {Object} convertors The convertor to handle.
     */
    handleConvertors: function(convertors, category)
    {
        // Nothing to do, no convertors on workspace richtext
    },
    
    /**
     * @private
     * Add a custom validation function to be called during inline editor validation ({@link Ext.form.field.Field#getErrors}).
     * @param {Function} validator The new validator to add. This function will have the following signature:
     * @param {Object} validator.value The current field value
     * @param {Boolean/String} validator.return
     * - True if the value is valid
     * - An error message if the value is invalid
     * @param {String} [category=""] The category where to register.
     */
    addValidator: function(validator, category)
    {
        category = category || "";
        
        this._validators[category] = this._validators[category] || []; 
        this._validators[category].push(validator);
    },
    
    /**
     * Validates the value among the existing validators for the inline editor
     * @param {String} value The current field value
     * @param {String} [category=""] The category to validates
     * @return {Boolean/String} validator.return
     *
     * - True if the value is valid
     * - An error message if the value is invalid
     */
    validates: function(value, category)
    {
        // Nothing to do, no validators on workspace richtext
        return true;
    },
    
    /**
     * @private
     * Add a custom convertor function to be called during inline editor conversion process getValue/setValue.
     * @param {Function} onGetContent The new function to add that will be called on getValue to convert the internal richtext structure to the external value
     * @param {Ametys.form.field.RichText} onGetContent.field The current richtext field
     * @param {tinymce.Editor} onGetContent.editor The current richtext editor
     * @param {Object} onGetContent.object The object of value to be modified
     * @param {Function} onSetContent The new function to add that will be called on setValue to convert the external value to the internal richtext structure 
     * @param {tinymce.Editor} onSetContent.editor The current richtext editor
     * @param {Object} onSetContent.object The object of value to be modified
     * @param {String} [category=""] The category where to register.
     */
    addConvertor: function(onGetContent, onSetContent, category)
    {
        category = category || "";
        
        if (onGetContent)
        {
            this._convertorsOnGetContent[category] = this._convertorsOnGetContent[category] || []; 
            this._convertorsOnGetContent[category].push(onGetContent);
        }

        if (onSetContent)
        {
            this._convertorsOnSetContent[category] = this._convertorsOnSetContent[category] || []; 
            this._convertorsOnSetContent[category].push(onSetContent);
        }
    },
    
    /**
     * Converts the value on get content
     * @param {Ametys.form.field.RichText} field The richtext to convert
     * @param {tinymce.Editor} editor The current richtext editor
     * @param {Object} object The object of value to be modified
     * @param {String} [category=""] The category to apply.
     */
    convertOnGetContent: function(field, editor, object, category)
    {
        // Nothing to do, no convertors on workspace richtext
    },
    
    /**
     * Converts the value on set content
     * @param {Ametys.form.field.RichText} field The richtext to convert
     * @param {tinymce.Editor} editor The current richtext editor
     * @param {Object} object The object of value to be modified
     * @param {String} [category=""] The category to apply.
     */
    convertOnSetContent: function(field, editor, object, category)
    {
        // Nothing to do, no convertors on workspace richtext
    },
    
    /**
     * @private
     * Check if RichTextConfiguration is initialized and throw an exception otherwise
     */
    checkIfInitialized: function()
    {
        RichTextConfigurationBase.checkIfInitialized();
    },
    
    /**
     * Get language code. 
     * @return {String} language code.
     */
    getLanguageCode: function()
    {
        return AmetysFront.getAppParameter("sitemapLanguage");
    },
    
    /**
     * Get debug mode. 
     * @return {Boolean} debug mode.
     */
    getDebugMode: function()
    {
        return AmetysFront.DEBUG_MODE;
    },
    
    /**
     * Get Rtl mode. 
     * @return {Boolean} Rtl mode.
     */
    getRtlMode: function()
    {
        // RTL mode is not supported since RUNTIME-3812
        return false;
    },
    
    /**
     * Get context path. 
     * @return {String} context path.
     */
    getContextPath: function()
    {
        return AmetysFront.CONTEXT_PATH;
    },
}

    
AmetysFront.ServerComm.callMethod({
    role: "org.ametys.core.ui.widgets.richtext.RichTextConfigurationExtensionPoint",
    methodName: "toJSON",
    parameters: [ AmetysFront.getAppParameters() ],
    callback: {
        handler: function(json) { RichTextConfigurationBase.setConfigurationFromServer(json, Ametys.form.widget.RichText.RichTextConfiguration); }
    },
    errorMessage: true
});
