/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';

/**
 * Pinia store for handling tags
 */
export const useTagsStore = defineStore('tags', {
  state: () => ({
        tags: [], // list of available tags
        canCreate: false // whether the user can create new tags
    }),
    actions:
    {
        /**
         * Load the tags for a given module id
         * @param {string} moduleId the id of the module to load tags for
         */
        async loadTags(moduleId)
        {
            try
            {
                let data = await callMethod({
                    role: "org.ametys.plugins.workspaces.tags.ProjectTagsDAO",
                    methodName: "getProjectTags",
                    parameters: [moduleId, AmetysFront.getAppParameter('projectName')]
                })

                data.tags.sort(function(c1,c2) {
                    if (AmetysFront.Utils.deemphasize(c1.text.toLowerCase()) == AmetysFront.Utils.deemphasize(c2.text.toLowerCase())) return 0;
                    else if (AmetysFront.Utils.deemphasize(c1.text.toLowerCase()) < AmetysFront.Utils.deemphasize(c2.text.toLowerCase())) return -1;
                    else return 1;
                });

                this.tags = data.tags;
                this.canCreate = data.canCreate;
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_LOADTAGS_FAIL,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_LOADTAGS_FAIL_TEXT,
                    details: e
                });
            }
        },

        /**
         * Add created tags to the list
         * @param {Array} tags the tags to add
         */
        async addTags(tags)
        {
            for (var i in tags)
            {
                this.tags.push(tags[i]);
            }
            this.tags.sort(function(c1,c2) {
                if (AmetysFront.Utils.deemphasize(c1.text.toLowerCase()) == AmetysFront.Utils.deemphasize(c2.text.toLowerCase())) return 0;
                else if (AmetysFront.Utils.deemphasize(c1.text.toLowerCase()) < AmetysFront.Utils.deemphasize(c2.text.toLowerCase())) return -1;
                else return 1;
            });
        }
    },
})

