/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';

/**
 * Pinia store for project keywords
 */
export const useProjectKeywordsStore = defineStore('projectKeywords', {
    state: () => (
    {
        keywords: [], // List of keywords
        canCreate: false // Whether the user can create keywords
    }),
    actions:
    {
        /**
         * Load the keywords list
         * @param {String} the zoneItemId of the catalog service, used to get allowed populations
         */
        async loadKeywords(zoneItemId) {
            try
            {
                let data = await callMethod({
                    role: "org.ametys.plugins.workspaces.keywords.KeywordsDAO",
                    methodName: "getKeywords",
                    parameters: [zoneItemId]
                })
                data.keywords.sort(function(c1,c2) {
                    if (c1.text == c2.text) return 0;
                    else if (c1.text < c2.text) return -1;
                    else return 1;
                });
                
                this.keywords = data.keywords;
                this.canCreate = data.canCreate;
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LOADKEYWORDS_FAIL,
                    text : i18n.PLUGINS_WORKSPACES_CATALOGUE_LOADKEYWORDS_FAIL_TEXT,
                    details: e
                });
    
            }
        },

        /**
         * Add keywords to the list
         * @param {String[]} keywords the keywords to add
         */
        addKeywords(keywords) {
            for (var i in keywords)
            {
                this.keywords.push(keywords[i]);
            }
        }
    }

})

