/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollectionFactory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityHelper;
import org.ametys.plugins.repository.activities.ActivityTypeExpression;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.activities.activitystream.ActivityStreamClientInteraction;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.MoveablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.service.Service;
import org.ametys.web.service.ServiceExtensionPoint;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinTemplate;
import org.ametys.web.skin.SkinsManager;

/**
 * Abstract class for {@link WorkspaceModule} implementation
 *
 */
public abstract class AbstractWorkspaceModule extends AbstractLogEnabled implements WorkspaceModule, Serviceable, Contextualizable, PluginAware
{

    /** Size value constants in case of size computation error */
    protected static final Long __SIZE_ERROR = -1L;
    
    /** Size value constants for inactive modules */
    protected static final Long __SIZE_INACTIVE = -2L;
    
    /** Project manager */
    protected ProjectManager _projectManager;
    /** Project right helper */
    protected ProjectRightHelper _projectRightHelper;
    /** User manager */
    protected UserManager _userManager;
    /** Ametys resolver */
    protected AmetysObjectResolver _resolver;
    /** The rights manager */
    protected RightManager _rightManager;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** The users manager */
    protected UserHelper _userHelper;
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;
    /** The skins manager. */
    protected SkinsManager _skinsManager;
    /** The page DAO */
    protected PageDAO _pageDAO;
    /** The avalon context */
    protected Context _context;
    /** The plugin name */
    protected String _pluginName;
    /** The services handler */
    protected ServiceExtensionPoint _serviceEP;
    /** The modules extension point */
    protected WorkspaceModuleExtensionPoint _modulesEP;
    /** The activity stream manager */
    protected ActivityStreamClientInteraction _activityStream;
    /** Workspaces helper */
    protected WorkspacesHelper _wokspacesHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectRightHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _wokspacesHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _skinsManager = (SkinsManager) manager.lookup(SkinsManager.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _serviceEP = (ServiceExtensionPoint) manager.lookup(ServiceExtensionPoint.ROLE);
        _modulesEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _activityStream = (ActivityStreamClientInteraction) manager.lookup(ActivityStreamClientInteraction.ROLE);
    }

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void deleteData(Project project)
    {
        // Delete module pages
        _deletePages(project);
        
        _internalDeleteData(project);
        
        // Delete root
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, false);
        if (moduleRoot != null)
        {
            moduleRoot.remove();
        }
        
        // Delete activities
        _deleteActivities(project);
    }
    
    @Override
    public void deactivateModule(Project project)
    {
        // Hide module pages
        _setPagesVisibility(project, false);
        
        _internalDeactivateModule(project);
    }
    
    @Override
    public void activateModule(Project project, Map<String, Object> additionalValues)
    {
        // create the resources root node
        getModuleRoot(project, true);
        _internalActivateModule(project, additionalValues);
        
        Site site = project.getSite();
        if (site != null)
        {
            for (Sitemap sitemap : site.getSitemaps())
            {
                initializeSitemap(project, sitemap);
            }
        }
        
        _setPagesVisibility(project, true);
    }
    
    @Override
    public void initializeSitemap(Project project, Sitemap sitemap)
    {
        ModifiablePage page = _createModulePage(project, sitemap, getModulePageName(), getModulePageTitle(), getModulePageTemplate());
        
        if (page != null)
        {
            page.tag("SECTION");
            _projectManager.tagProjectPage(page, getModuleRoot(project, true));
            
            initializeModulePage(page);
            
            page.saveChanges();
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_PAGE, page);
            _observationManager.notify(new Event(ObservationConstants.EVENT_PAGE_ADDED, _currentUserProvider.getUser(), eventParams));
        }
    }
    
    @Override
    public String getModuleUrl(Project project)
    {
        Optional<String> url = _projectManager.getModulePages(project, this).stream()
            .findFirst()
            .map(page -> ResolveURIComponent.resolve("page", page.getId()));
        
        if (url.isPresent())
        {
            return url.get();
        }
        else
        {
            // No page found
            return null;
        }
    }
    
    /**
     * Create a new page if not already exists
     * @param project The module project
     * @param sitemap The sitemap where the page will be created
     * @param name The page's name
     * @param pageTitle The page's title as i18nizable text
     * @param skinTemplate The template from the skin to apply on the page
     * @return the created page or <code>null</code> if page already exists
     */
    protected ModifiablePage _createModulePage(Project project, Sitemap sitemap, String name, I18nizableText pageTitle, String skinTemplate)
    {
        if (!sitemap.hasChild(name))
        {
            ModifiablePage page = sitemap.createChild(name, "ametys:defaultPage");
            
            // Title should not be missing, but just in case if the i18n message or the whole catalog does not exists in the requested language
            // to prevent a non-user-friendly error and still generate the project workspace.
            page.setTitle(StringUtils.defaultIfEmpty(_i18nUtils.translate(pageTitle, sitemap.getName()), "Missing title"));
            page.setType(PageType.NODE);
            page.setSiteName(sitemap.getSiteName());
            page.setSitemapName(sitemap.getName());
            
            Site site = page.getSite();
            Skin skin = _skinsManager.getSkin(site.getSkinId());
            
            if (skinTemplate != null)
            {
                SkinTemplate template = skin.getTemplate(skinTemplate);
                if (template != null)
                {
                    // Set the type and template.
                    page.setType(PageType.CONTAINER);
                    page.setTemplate(skinTemplate);
                }
                else
                {
                    getLogger().error(String.format(
                            "The project workspace  '%s' was created with the skin '%s'  which doesn't possess the mandatory template '%s'.\nThe '%s' page of the project workspace could not be initialized.",
                            site.getName(), site.getSkinId(), skinTemplate, page.getName()));
                }
            }
            
            sitemap.saveChanges();

            // Move module page to ensure pages order
            for (WorkspaceModule otherModule : _modulesEP.getModules())
            {
                if (otherModule.compareTo(this) > 0)
                {
                    Set<Page> modulePages = _projectManager.getModulePages(project, otherModule);
                    if (!modulePages.isEmpty())
                    {
                        ((MoveablePage) page).orderBefore(modulePages.iterator().next());
                        break;
                    }
                }
            }

            sitemap.saveChanges();

            return page;
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Change the visibility of module pages if needed
     * @param project The project
     * @param visible visible <code>true</code> to set pages as visible, <code>false</code> otherwise
     */
    protected void _setPagesVisibility(Project project, boolean visible)
    {
        List<String> modulePageIds = _getModulePages(project)
                .stream()
                .filter(p -> !"index".equals(p.getPathInSitemap()) && (visible && !p.isVisible() || !visible && p.isVisible()))
                .map(Page::getId)
                .collect(Collectors.toList());
        
        _pageDAO.setVisibility(modulePageIds, visible);
    }
    
    /**
     * Delete the module pages and their related contents
     * @param project The project
     */
    protected void _deletePages(Project project)
    {
        List<Page> modulePages = _getModulePages(project);
        
        for (Page page : modulePages)
        {
            _pageDAO.deletePage((ModifiablePage) page, true);
        }
    }
    
    /**
     * Get the module pages
     * @param project the project
     * @return the module pages
     */
    protected List<Page> _getModulePages(Project project)
    {
        String modulePageName = getModulePageName();
        List<Page> pages = new ArrayList<>();
        Site site = project.getSite();
        if (site != null)
        {
            for (Sitemap sitemap : site.getSitemaps())
            {
                if (sitemap.hasChild(modulePageName))
                {
                    pages.add(sitemap.getChild(modulePageName));
                }
            }
        }
        
        return pages;
    }

    /**
     * Delete all activities related to this module
     * @param project The project
     */
    protected void _deleteActivities(Project project)
    {
        Expression projectExpression = new StringExpression(AbstractWorkspacesActivityType.PROJECT_NAME, Operator.EQ, project.getName());
        List<Expression> eventTypeExpressions = new ArrayList<>();
        for (String eventType: getAllowedEventTypes())
        {
            eventTypeExpressions.add(new ActivityTypeExpression(Operator.EQ, eventType));
        }
        Expression moduleActivityExpression = new AndExpression(projectExpression, new OrExpression((Expression[]) eventTypeExpressions.toArray()));
        String query = ActivityHelper.getActivityXPathQuery(moduleActivityExpression);
        AmetysObjectIterable<Activity> activities = _resolver.query(query);
        for (Activity activity : activities)
        {
            activity.remove();
        }
    }
    
    /**
     * Get the default value of the XSLT parameter of the given service.
     * @param serviceId the service ID.
     * @return the default XSLT parameter value.
     */
    protected String _getDefaultXslt(String serviceId)
    {
        Service service = _serviceEP.hasExtension(serviceId) ? _serviceEP.getExtension(serviceId) : null;
        if (service != null)
        {
            @SuppressWarnings("unchecked")
            ElementDefinition<String> xsltParameterDefinition = (ElementDefinition<String>) service.getParameters().get("xslt");
            
            if (xsltParameterDefinition != null)
            {
                return xsltParameterDefinition.getDefaultValue();
            }
        }
        
        return StringUtils.EMPTY;
    }
    
    /**
     * Returns the module page's name
     * @return The module page's name
     */
    protected abstract String getModulePageName();
    
    /**
     * Returns the module page's title as i18n
     * @return The module page's title
     */
    protected abstract I18nizableText getModulePageTitle();
    
    /**
     * Returns the template to use for module's page. Can be null if the page should be a node page
     * @return The template
     */
    protected String getModulePageTemplate()
    {
        return ProjectConstants.PROJECT_TEMPLATE;
    }
    
    /**
     * Initialize the module page
     * @param modulePage The module page
     */
    protected abstract void initializeModulePage(ModifiablePage modulePage);
    
    /**
     * Internal process when module is deactivated
     * @param project The project
     */
    protected void _internalDeactivateModule(Project project)
    {
        // Empty
    }
    
    /**
     * Internal process to delete data
     * @param project The project
     */
    protected void _internalDeleteData(Project project)
    {
        // Empty
    }
    
    /**
     * Internal process when module is activated
     * @param project The project
     * @param additionalValues A list of optional additional values. Accepted values are : description, mailingList, inscriptionStatus, defaultProfile, tags, categoryTags, keywords and language
     */
    protected void _internalActivateModule(Project project, Map<String, Object> additionalValues)
    {
        // Empty
    }
    
    /**
     * Utility method to get or create an ametys object
     * @param <A> A sub class of AmetysObject
     * @param parent The parent object
     * @param name The ametys object name
     * @param type The ametys object type
     * @param create True to create the object if it does not exist
     * @return ametys object
     * @throws AmetysRepositoryException if an repository error occurs
     */
    protected <A extends AmetysObject> A _getAmetysObject(ModifiableTraversableAmetysObject parent, String name, String type, boolean create) throws AmetysRepositoryException
    {
        A object = null;
        
        if (parent.hasChild(name))
        {
            object = parent.getChild(name);
        }
        else if (create)
        {
            object = parent.createChild(name, type);
            parent.saveChanges();
        }
        
        return object;
    }

    @Override
    public Map<String, Object> getStatistics(Project project)
    {
        Map<String, Object> statistics = new HashMap<>();

        if (ArrayUtils.contains(project.getModules(), getId()))
        {
            statistics.put(_getModuleLastActivityKey(), _getModuleLastActivity(project));
            statistics.put(_getModuleAtivateKey(), true);
            statistics.put(getModuleSizeKey(), _getModuleSize(project));
            statistics.putAll(_getInternalStatistics(project, true));
        }
        else
        {
            statistics.put(_getModuleAtivateKey(), false);
            statistics.putAll(_getInternalStatistics(project, false));
            // Use -2 as default empty value, so the sort in columns can work. It will be replaced by empty value in the renderer.
            statistics.put(getModuleSizeKey(), __SIZE_INACTIVE);
        }
        
        return statistics;
    }

    /**
     * Get the internal statistics of the module
     * @param project The project
     * @param isActive true if module is active
     * @return a map of internal statistics
     */
    protected Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    {
        return Map.of();
    }
    
    @Override
    public List<StatisticColumn> getStatisticModel()
    {
        List<StatisticColumn> statisticHeaders = new ArrayList<>();

        if (_showActivatedStatus())
        {
            statisticHeaders.add(new StatisticColumn(_getModuleAtivateKey(), getModuleTitle())
                    .withGroup(GROUP_HEADER_ACTIVATED_ID)
                    .withType(StatisticsColumnType.BOOLEAN));
        }
        if (_showLastActivity())
        {
            statisticHeaders.add(new StatisticColumn(_getModuleLastActivityKey(), getModuleTitle())
                    .withGroup(GROUP_HEADER_LAST_ACTIVITY_ID)
                    .withType(StatisticsColumnType.DATE));
        }
        
        if (_showModuleSize())
        {
            statisticHeaders.add(new StatisticColumn(getModuleSizeKey(), getModuleTitle())
                    .withType(StatisticsColumnType.LONG)
                    .withGroup(GROUP_HEADER_SIZE_ID)
                    .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderSize")
                    .isHidden(true));
        }
        
        statisticHeaders.addAll(_getInternalStatisticModel());
        
        return statisticHeaders;
    }
    
    /**
     * Get the headers of statistics
     * @return a list of statistics headers
     */
    protected List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of();
    }

    @Override
    public String getModuleSizeKey()
    {
        return getModuleName() + "$size";
    }
    
    private String _getModuleAtivateKey()
    {
        return getModuleName() + "$activated";
    }
    
    private String _getModuleLastActivityKey()
    {
        return getModuleName() + "$lastActivity";
    }

    /**
     * Get the size of module in bytes
     * @param project The project
     * @return the size of module in bytes
     */
    protected long _getModuleSize(Project project)
    {
        return 0;
    }

    /**
     * Check if activated status should be shown or not
     * @return true if activated status should be shown
     */
    protected boolean _showActivatedStatus()
    {
        return true;
    }

    /**
     * Check if module size should be shown or not
     * @return true if module size should be shown
     */
    protected boolean _showModuleSize()
    {
        return false;
    }

    public Set<String> getAllEventTypes()
    {
        return getAllowedEventTypes();
    }

    /**
     * Check if the last activity should be shown or not
     * @return true if last activity should be shown
     */
    protected boolean _showLastActivity()
    {
        return getAllEventTypes().size() != 0;
    }
    
    /**
     * Get the size of module in bytes
     * @param project The project
     * @return the size of module in bytes
     */
    protected ZonedDateTime _getModuleLastActivity(Project project)
    {
        return _activityStream.getDateOfLastActivityByActivityType(project.getName(), getAllowedEventTypes());
    }
    
    public ModifiableResourceCollection getModuleRoot(Project project, boolean create)
    {
        try
        {
            ExplorerNode projectRootNode = project.getExplorerRootNode();
            
            if (projectRootNode instanceof ModifiableResourceCollection mProjectRootNode)
            {
                return _getAmetysObject(mProjectRootNode, getModuleName(), JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE, create);
            }
            else
            {
                throw new IllegalArgumentException("Root module '" + projectRootNode.getPath() + "' is not modifiable");
            }
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the " + getModuleName() + " root node.", e);
        }
    }
}
