/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.calendars.events;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.workspaces.calendars.AbstractCalendarDAO;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarDAO;
import org.ametys.plugins.workspaces.calendars.task.TaskCalendarEvent;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;

/**
 * Helper to convert events to JSON
 */
public class CalendarEventJSONHelper extends AbstractCalendarDAO
{
    /** Avalon Role */
    public static final String ROLE = CalendarEventJSONHelper.class.getName();

    private static final String __HOUR_PATTERN = "yyyyMMdd'T'HHmmssXXX";
    
//    private static final String __FULL_DAY_PATTERN = "uuuuMMdd";
    
    /** Calendar DAO */
    protected CalendarDAO _calendarDAO;

    private TasksWorkspaceModule _taskModule;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _calendarDAO = (CalendarDAO) manager.lookup(CalendarDAO.ROLE);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _taskModule = moduleManagerEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
    }
        
    /**
     * Get event info for a specific occurrence
     * @param event The event
     * @param occurrenceDate the occurrence
     * @param fullInfo true to include full info (rights, parent id, etc...)
     * @return the event data in a map
     */
    public Map<String, Object> eventAsJsonWithOccurrence(CalendarEvent event, ZonedDateTime occurrenceDate, boolean fullInfo)
    {
        Map<String, Object> eventData = eventAsJson(event, fullInfo, false);
        
        Optional<CalendarEventOccurrence> optionalEvent = event.getFirstOccurrence(occurrenceDate);
        if (optionalEvent.isPresent())
        {
            eventData.putAll(optionalEvent.get().toJSON());
        }
        
        return eventData;
    }
    
    /**
     * Get event info
     * @param event The event
     * @param fullInfo true to include full info (rights, parent id, etc...)
     * @param startDate The start date.
     * @param endDate The end date.
     * @return the event data in a map
     */
    public Map<String, Object> eventAsJsonWithOccurrences(CalendarEvent event, boolean fullInfo, ZonedDateTime startDate, ZonedDateTime endDate)
    {
        Map<String, Object> eventData = eventAsJson(event, false, false);

        List<CalendarEventOccurrence> occurences = event.getOccurrences(startDate, endDate);
        
        List<Object> occurrencesDataList = new ArrayList<>();
        eventData.put("occurrences", occurrencesDataList);
        
        for (CalendarEventOccurrence occurence : occurences)
        {
            occurrencesDataList.add(occurence.toJSON());
        }
        return eventData;
    }
    
    /**
     * Get event info
     * @param event The event
     * @param fullInfo true to include full info (rights, parent id, etc...)
     * @param useICSFormat true to use ICS Format for dates
     * @return the event data in a map
     */
    public Map<String, Object> eventAsJson(CalendarEvent event, boolean fullInfo, boolean useICSFormat)
    {
        
        Calendar calendar = event.getCalendar();
        Map<String, Object> result = new HashMap<>();
        
        result.put("id", event.getId());
        result.put("calendarId", calendar.getId());
        result.put("color", calendar.getColor());
        
        result.put("title", event.getTitle());
        result.put("description", event.getDescription());
        
        boolean fullDay = event.getFullDay();
        
        result.put("fullDay", fullDay);
        result.put("recurrenceType", event.getRecurrenceType().toString());
        
        result.put("location", event.getLocation());
        result.put("keywords", event.getTags());
        
        ZonedDateTime untilDate = event.getRepeatUntil();
        if (untilDate != null)
        {
            if (useICSFormat)
            {
                // iCalendar/ICS handle until date slightly differently, so we have to add one day to include the last occurrence
                untilDate = untilDate.plusDays(1).minusSeconds(1);
            }
            
            result.put("untilDate", formatDate(untilDate, useICSFormat, fullDay));
        }
        
        ZonedDateTime startDateEvent = event.getStartDate();
        ZonedDateTime endDateEvent = event.getEndDate();
        
        if (event.getFullDay())
        {
            result.put("endDateNextDay", formatDate(endDateEvent.plusDays(1), useICSFormat, fullDay));
        }

        result.put("startDate", formatDate(startDateEvent, useICSFormat, fullDay));
        
        if (event.getFullDay() && useICSFormat)
        {
            result.put("endDate", formatDate(endDateEvent.plusDays(1), useICSFormat, fullDay));
        }
        else
        {
            result.put("endDate", formatDate(endDateEvent, useICSFormat, fullDay));
        }
        
        //excluded occurences
        List<ZonedDateTime> excludedOccurences = event.getExcludedOccurences();
        if (excludedOccurences != null && !excludedOccurences.isEmpty())
        {
            List<String> excludedOccurencesStrings = new ArrayList<>();
            for (ZonedDateTime excludedOccurence : excludedOccurences)
            {
                excludedOccurencesStrings.add(formatDate(excludedOccurence, useICSFormat, true));
            }
            result.put("excludedDates", excludedOccurencesStrings);
        }

        // creator
        UserIdentity creatorIdentity = event.getCreator();
        User creator = _userManager.getUser(creatorIdentity);
        
        result.put("creator", creatorIdentity);
        result.put("creatorFullName", creator != null ? creator.getFullName() : creatorIdentity.getLogin());

        UserIdentity user = _currentUserProvider.getUser();
        result.put("isCreator", creatorIdentity.equals(user));
        result.put("creationDate", formatDate(event.getCreationDate(), useICSFormat, false));
        
        // last modification
        UserIdentity contributorIdentity = event.getLastContributor();
        User contributor = _userManager.getUser(contributorIdentity);
        
        result.put("contributor", contributorIdentity);
        result.put("contributorFullName", contributor != null ? contributor.getFullName() : contributorIdentity.getLogin());
        result.put("lastModified", formatDate(event.getLastModified(), useICSFormat, false));
        
        if (fullInfo)
        {
            result.putAll(_eventAsJsonFullInfo(event));
        }

        result.put("calendar", _calendarDAO.getCalendarProperties(calendar));
        
        // tags and places are expected by the client (respectively keywords and location on the server side)
        result.put("tags", event.getTags());
        
        String location = StringUtils.defaultString(event.getLocation());
        result.put("location", location);
        result.put("places", Stream.of(location.split(",")).filter(StringUtils::isNotEmpty).collect(Collectors.toList()));
        
        // add event rights
        result.put("rights", _extractEventRightData(event));
        
        result.put("resourceIds", event.getResources());
        
        result.put("isModifiable", event instanceof ModifiableCalendarEvent);
        if (event instanceof TaskCalendarEvent taskCalendarEvent)
        {
            Task task = taskCalendarEvent.getTask();
            Project project = _projectManager.getParentProject(task);
            result.put("taskURL",  _taskModule.getTaskUri(project, task.getId()));
           
        }

        return result;
    }
    
    private String formatDate(ZonedDateTime date, boolean useICSFormat, boolean fullDay)
    {
        if (useICSFormat)
        {
            if (fullDay)
            {
                // FIXME : commented because currently, dates are badly stored, waiting for a data migration to use it back
            //  return DateUtils.zonedDateTimeToString(date, date.getZone(), __FULL_DAY_PATTERN);
                return DateUtils.zonedDateTimeToString(date);
            }
            else
            {
                return DateUtils.zonedDateTimeToString(date, date.getZone(), __HOUR_PATTERN);
            }
        }
        else
        {
            return DateUtils.zonedDateTimeToString(date);
        }
    }
    
    /**
     * Retrieves the event additional info (rights, parent id, etc...)
     * @param event The event
     * @return the event additional info (rights, parent id, etc...) in a map
     */
    protected Map<String, Object> _eventAsJsonFullInfo(CalendarEvent event)
    {
        Map<String, Object> result = new HashMap<>();
        
        ExplorerNode explorerNode = event.getParent();
        ExplorerNode root = explorerNode;
        while (true)
        {
            if (root.getParent() instanceof ExplorerNode)
            {
                root = root.getParent();
            }
            else
            {
                break;
            }
        }
        result.put("rootId", root.getId());
        result.put("parentId", explorerNode.getId());
        result.put("name", event.getName());
        result.put("path", explorerNode.getExplorerPath());
        
        result.put("rights", _getUserRights(explorerNode));
        
        return result;
    }
    
    /**
     * Internal method to extract the data concerning the right of the current user for an event
     * @param event The event
     * @return The map of right data. Keys are the rights id, and values indicates whether the current user has the right or not.
     */
    protected  Map<String, Object> _extractEventRightData(CalendarEvent event)
    {
        Map<String, Object> rightsData = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        Calendar calendar = event.getCalendar();
        
        rightsData.put("edit", event instanceof ModifiableCalendarEvent && _rightManager.hasRight(user, AbstractCalendarDAO.RIGHTS_EVENT_EDIT, calendar) == RightResult.RIGHT_ALLOW);
        rightsData.put("delete", event instanceof ModifiableCalendarEvent && _rightManager.hasRight(user, AbstractCalendarDAO.RIGHTS_EVENT_DELETE, calendar) == RightResult.RIGHT_ALLOW);
        rightsData.put("delete-own", event instanceof ModifiableCalendarEvent && _rightManager.hasRight(user, AbstractCalendarDAO.RIGHTS_EVENT_DELETE_OWN, calendar) == RightResult.RIGHT_ALLOW);
        
        return rightsData;
    }
    
    /**
     * Get the user rights on the resource collection
     * @param node The explorer node
     * @return The user's rights
     */
    protected Set<String> _getUserRights(ExplorerNode node)
    {
        return _rightManager.getUserRights(_currentUserProvider.getUser(), node);
    }
    
}
