/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.generators;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Filtered events generator for the simple service.
 */
public class FilteredEventsGenerator extends ServiceableGenerator
{
    /** The calendar manager */
    private CalendarWorkspaceModule _calendarModule;
    private ProjectsCatalogueManager _projectCatalogManager;
    private AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _calendarModule = moduleManagerEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        _projectCatalogManager = (ProjectsCatalogueManager)  serviceManager.lookup(ProjectsCatalogueManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        int months = parameters.getParameterAsInteger("search-period", 3);
        int max = parameters.getParameterAsInteger("max-results", Integer.MAX_VALUE);
        
        List<String> calendarIds = null;
        List<String> tagIds = null;
        
        XMLUtils.startElement(contentHandler, "events");
        
        _calendarModule.getUpcomingEvents(months, max, calendarIds, tagIds)
            .stream()
            .forEach(this::_saxEvent);
        
        XMLUtils.endElement(contentHandler, "events");

        contentHandler.endDocument();
    }
    
    /**
     * SAX necessary event properties for the simple filtered event service.
     * @param eventData map containing the event data
     */
    protected void _saxEvent(Map<String, Object> eventData)
    {
        try
        {
            boolean calendarHasViewRight = BooleanUtils.isTrue((Boolean) eventData.get("calendarHasViewRight"));
            boolean calendarIsPublic = BooleanUtils.isTrue((Boolean) eventData.get("calendarIsPublic"));
            
            AttributesImpl attrs = new AttributesImpl();
            
            attrs.addCDATAAttribute("id", (String) eventData.get("id"));
            attrs.addCDATAAttribute("calendarHasViewRight", Boolean.toString(calendarHasViewRight));
            attrs.addCDATAAttribute("calendarIsPublic", Boolean.toString(calendarIsPublic));
            XMLUtils.startElement(contentHandler, "event", attrs);
            
            // project 
            attrs.clear();
            String projectId = (String) eventData.get("projectId");
            attrs.addCDATAAttribute("id", (String) eventData.get(projectId));
            XMLUtils.createElement(contentHandler, "project", (String) eventData.get("projectTitle"));
            
            Project project = _resolver.resolveById(projectId);
            _projectCatalogManager.saxCategory(contentHandler, project, "projectCategory");
            
            @SuppressWarnings("unchecked")
            Map<String, Object> calendarData = (Map<String, Object>) eventData.get("calendar");
            XMLUtils.createElement(contentHandler, "calendar", (String) calendarData.get("title"));
            
            XMLUtils.createElement(contentHandler, "title", (String) eventData.get("title"));
            
            String startDate = (String) eventData.get("startDate");
            XMLUtils.createElement(contentHandler, "startDate", startDate);
            
            // Link to calendar module is provided only is the user has the right to view the calendar
            if (calendarHasViewRight)
            {
                String eventUrl = (String) eventData.get("eventUrl");
                if (StringUtils.isNotEmpty(eventUrl))
                {
                    XMLUtils.createElement(contentHandler, "eventUrl", eventUrl);
                }
            }
            
            XMLUtils.endElement(contentHandler, "event");
        }
        catch (SAXException e)
        {
            throw new RuntimeException("An error occurred while gathering the events' information.");
        }
    }
}
