/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.categories;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.tag.jcr.AbstractJCRTagProvider;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Class representing a project jcr category provider. <br>
 */
public class CategoryJCRProvider extends AbstractJCRTagProvider<Category>
{
    /** Constant for plugin node name */ 
    public static final String PROJECT_PLUGIN_NODE_NAME = "workspaces";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The projet tag provider extension point */
    protected CategoryProviderExtensionPoint _projectTagProviderEP;
    
    /** The colors component */
    protected CategoryColorsComponent _colorsComponent;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _projectTagProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _colorsComponent = (CategoryColorsComponent) smanager.lookup(CategoryColorsComponent.ROLE);
    }
    
    @Override
    protected void _fillCache(TraversableAmetysObject parentTagNode, Category parentTag, Map<String, Category> cache) throws RepositoryException
    {
        for (AmetysObject child : parentTagNode.getChildren())
        {
            if (child instanceof CategoryJCR)
            {
                CategoryJCR jcrTag = (CategoryJCR) child;
                Category tag = new Category(jcrTag.getId(), jcrTag.getName(), jcrTag.getColor(), _colorsComponent, parentTag, new I18nizableText(jcrTag.getTitle()), new I18nizableText(jcrTag.getDescription()));
                cache.put(child.getName(), tag);
                
                if (parentTag != null)
                {
                    parentTag.addTag(tag);
                }
                
                _fillCache(jcrTag, tag, cache);
            }
        }
    }
    
    @Override
    protected ModifiableTraversableAmetysObject _getRootNode(Map<String, Object> contextualParameters, Session session) throws RepositoryException
    {
        try
        {
            ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins", session);
            
            ModifiableTraversableAmetysObject pluginNode = _getOrCreateNode(pluginsNode, PROJECT_PLUGIN_NODE_NAME, "ametys:unstructured");
            
            return _getOrCreateNode(pluginNode, _projectTagProviderEP.getTagsNodeName(), _projectTagProviderEP.getTagsNodeType());
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the JCR tags root node", e);
        }
    }
    
    @Override
    public Collection<Category> getTags(String tagID, Map<String, Object> contextualParameters)
    {
        Category tag = getTag(tagID, contextualParameters);
        return tag != null ? tag.getTags().values() : null;
    }
    
    @Override
    public List<String> getCSSUrls(Map<String, Object> contextualParameters)
    {
        List<String> cssUrls = super.getCSSUrls(contextualParameters);
        cssUrls.add("/plugins/workspaces/project-categories.min.css");
        
        return cssUrls;
    }
}
