/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.dav;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.runtime.config.Config;

/**
 * Reader for WebDAV LOCK method
 */
public class WebdavLockGenerator extends AbstractGenerator implements Serviceable
{
    private CurrentUserProvider _currentUserProvider;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        ModifiableResource resource = (ModifiableResource) request.getAttribute("resource");
        
        Response response = ObjectModelHelper.getResponse(objectModel);
        response.setHeader("Lock-Token", "<" + resource.getId() + ">");
        response.setHeader("Content-Type", "application/xml; charset=utf-8");
        
        contentHandler.startDocument();
        
        contentHandler.startPrefixMapping("d", "DAV:");

        XMLUtils.startElement(contentHandler, "d:prop");
        XMLUtils.startElement(contentHandler, "d:lockdiscovery");
        XMLUtils.startElement(contentHandler, "d:activelock");
        
        // <D:locktype><D:write/></D:locktype> 
        XMLUtils.startElement(contentHandler, "d:locktype");
        XMLUtils.createElement(contentHandler, "d:write");
        XMLUtils.endElement(contentHandler, "d:locktype");
        
        // <D:lockscope><D:exclusive/></D:lockscope>
        XMLUtils.startElement(contentHandler, "d:lockscope");
        XMLUtils.createElement(contentHandler, "d:exclusive");
        XMLUtils.endElement(contentHandler, "d:lockscope");
        
        // <D:depth>infinity</D:depth>
        XMLUtils.createElement(contentHandler, "d:depth", "infinity");
        
        XMLUtils.startElement(contentHandler, "d:locktoken");
        XMLUtils.createElement(contentHandler, "d:href", resource.getId());
        XMLUtils.endElement(contentHandler, "d:locktoken");
        
        boolean unlockActive = Config.getInstance().getValue("content.unlocktimer.activate");
        long unlockTimer = Config.getInstance().getValue("content.unlocktimer.period");
        
        XMLUtils.createElement(contentHandler, "d:timeout", unlockActive ? "Second-" + String.valueOf(3600 * unlockTimer) : "Infinite");

        XMLUtils.createElement(contentHandler, "d:owner", UserIdentity.userIdentityToString(_currentUserProvider.getUser()));
        
        String href = ResolveURIComponent.resolve("webdav-project-resource", resource.getId(), false, true);
        XMLUtils.startElement(contentHandler, "d:lockroot");
        XMLUtils.createElement(contentHandler, "d:href", href);
        XMLUtils.endElement(contentHandler, "d:lockroot");
        
        XMLUtils.endElement(contentHandler, "d:activelock");
        XMLUtils.endElement(contentHandler, "d:lockdiscovery");
        XMLUtils.endElement(contentHandler, "d:prop");
        
        contentHandler.endDocument();
    }
}
