/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.documents.onlyoffice;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Manager for key/token used in OnlyOffice
 */
public class OnlyOfficeKeyManager extends AbstractLogEnabled implements Component, Initializable
{
    /** The Avalon role */
    public static final String ROLE = OnlyOfficeKeyManager.class.getName();
    
    private Map<String, String> _onlyOfficeKeyStorage;
    
    public void initialize() throws Exception
    {
        _onlyOfficeKeyStorage = new ConcurrentHashMap<>();
    }
    
    /**
     * Get the OnlyOffice key associated with the resource
     * If a key exist for the resourceId, return it and increment _nbHolders
     * If no key exist for the resourceId, create a new key, put at 1 the _nbHolders and return the key
     * @param resourceId id of the resource to look
     * @return the OnlyOffice key
     */
    public synchronized String getKey(String resourceId)
    {
        String tokenKey = _resourceIdToTokenKey(resourceId);
        return _onlyOfficeKeyStorage.computeIfAbsent(tokenKey, this::createKey);
    }
    
    /**
     * Create the OnlyOffice key
     * @param tokenKey id of the resource associate to the key
     * @return the new key
     */
    protected String createKey(String tokenKey)
    {
        String currentTime = new SimpleDateFormat("yyyy.MM.dd.HH.mm.ss").format(new Date());
        return tokenKey + "." + currentTime;
    }
    
    /**
     * Remove OnlyOffice key
     * @param resourceId id of the resource
     */
    public void removeKey(String resourceId)
    {
        String tokenKey = _resourceIdToTokenKey(resourceId);
        _onlyOfficeKeyStorage.remove(tokenKey);
    }
    
    private String _resourceIdToTokenKey(String ressourceId)
    {
        return StringUtils.substringAfter(ressourceId, "://");
    }
}
