/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.initialization;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;

import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.filereloader.FileReloader;
import org.ametys.core.util.filereloader.FileReloaderUtils;
import org.ametys.plugins.workspaces.PagePopulator;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.ModifiableSitemapElement;
import org.ametys.web.repository.page.Page;

/**
 * Implementation of {@link PageInitializer} using a configuration to define the page and its content.
 * 
 * By default, priority is set to 100
 */
public class DefaultStaticPageInitializer extends AbstractLogEnabled implements PageInitializer, Configurable, Serviceable
{
    /** the current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** the observation manager */
    protected ObservationManager _observationManager;

    private PagePopulator _pagePopulator;
    private Map<String, Configuration> _pageModels = new HashMap<>();
    private int _priority;
    private String _path;
    private FileReloaderUtils _fileReloader;

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _path = configuration.getChild("model").getAttribute("path");
        _priority = configuration.getChild("priority", true).getValueAsInteger(100);
    }

    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _pagePopulator = (PagePopulator) manager.lookup(PagePopulator.ROLE);
        _fileReloader = (FileReloaderUtils) manager.lookup(FileReloaderUtils.ROLE);
    }
    
    public int getPriority()
    {
        return _priority;
    }
    
    public Optional<? extends Page> createPage(ModifiableSitemapElement parent)
    {
        Configuration model = _loadPageModel(parent.getSite().getSkinId());
        // null if the skin has no model for the given path
        if (model != null)
        {
            try
            {
                return _pagePopulator.initPage(parent, model);
            }
            catch (ConfigurationException e)
            {
                getLogger().error("An error occurred during the page initialization because the configuration was invalid.", e);
            }
            catch (Exception e)
            {
                getLogger().error("An unexpected error occurred during the page initialization.", e);
            }
        }
        return Optional.empty();
    }

    private Configuration _loadPageModel(String skinId)
    {
        try
        {
            String skinPath = StringUtils.replace(_path, "{skinId}", skinId);
            _fileReloader.updateFile(skinPath, new PageModelReloader(skinPath, this));
            
            // The reloader is in charge of storing and updating the model
            return _pageModels.get(skinPath);
        }
        catch (Exception e)
        {
            throw new IllegalStateException("No configuration found at path '" + _path + "' for skin '" + skinId + "'.", e);
        }
    }

    /**
     * Store the page model located at a given path
     * @param path the path to retrieve the model
     * @param cfg the page model
     */
    public void setModel(String path, Configuration cfg)
    {
        _pageModels.put(path, cfg);
    }
    
    /**
     *  File reloader for the page model configuration
     * @param path the file path.
     * @param initializer the initializer component
     */
    protected static record PageModelReloader(String path, DefaultStaticPageInitializer initializer) implements FileReloader {
        public void updateFile(String sourceUrl, Source source, InputStream is) throws Exception
        {
            if (is != null)
            {
                initializer.setModel(path, new DefaultConfigurationBuilder().build(is, source.getURI()));
            }
        }

        public String getId(String sourceUrl)
        {
            return PageModelReloader.class.getName() + "#" + path;
        }
    }

}
