/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.tag.AbstractTagProviderExtensionPoint;
import org.ametys.cms.tag.TagProvider;
import org.ametys.core.right.Profile;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.runtime.config.Config;

/**
 * Generator for the vuejs catalog service 
 */
public class ProjectsCatalogueGenerator extends ServiceableGenerator
{
    /** Right profiles manager */
    protected RightProfilesDAO _rightProfilesDao;
    /** The tag provider extension point */
    protected AbstractTagProviderExtensionPoint<Category> _tagProviderExtPt;
    
    private WorkspaceModuleExtensionPoint _modulesExtensionPoint;
    private CategoryHelper _categoryHelper;

    @SuppressWarnings("unchecked")
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _rightProfilesDao = (RightProfilesDAO) smanager.lookup(RightProfilesDAO.ROLE);
        _tagProviderExtPt = (AbstractTagProviderExtensionPoint<Category>) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _categoryHelper = (CategoryHelper) smanager.lookup(CategoryHelper.ROLE);
        _modulesExtensionPoint  = (WorkspaceModuleExtensionPoint) smanager.lookup(WorkspaceModuleExtensionPoint.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "catalog");
        
        _saxProfiles();
        _saxCategories();
        _saxModules();
        
        XMLUtils.endElement(contentHandler, "catalog");
        contentHandler.endDocument();
    }

    private void _saxModules() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "modules");
        
        for (WorkspaceModule workspaceModule : _modulesExtensionPoint.getModules())
        {
            if (!ProjectsCatalogueManager.DEFAULT_MODULES.contains(workspaceModule.getId()))
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", workspaceModule.getId());
                if (workspaceModule.isUnactivatedByDefault())
                {
                    attrs.addCDATAAttribute("unselected", "true");
                }
                XMLUtils.startElement(contentHandler, "module", attrs);
                
                workspaceModule.getModuleTitle().toSAX(contentHandler, "title");
                workspaceModule.getModuleDescription().toSAX(contentHandler, "description");
                
                XMLUtils.endElement(contentHandler, "module");
            }
        }
        
        XMLUtils.endElement(contentHandler, "modules");
    }

    private void _saxCategories() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "categories");
        
        for (String id : _tagProviderExtPt.getExtensionsIds())
        {
            TagProvider<? extends Category> categoryProvider = _tagProviderExtPt.getExtension(id);
            Map<String, Category> tags = categoryProvider.getTags(MapUtils.EMPTY_MAP);
            for (Category category : tags.values())
            {
                _saxCategory(category);
            }
        }

        
        XMLUtils.endElement(contentHandler, "categories");
    }

    private void _saxCategory(Category category) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "category");
        
        XMLUtils.createElement(contentHandler, "name", category.getName());
        category.getTitle().toSAX(contentHandler, "title");
        
        Map<String, String> categoryColor = _categoryHelper.getCategoryColor(category);
        XMLUtils.createElement(contentHandler, "color", categoryColor.get("main"));
        
        Map<String, Category> tags = category.getTags();
        if (!tags.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "children");
            for (Category tag : tags.values())
            {
                _saxCategory(tag);
            }
            XMLUtils.endElement(contentHandler, "children");
        }
        
        XMLUtils.endElement(contentHandler, "category");
    }

    private void _saxProfiles() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "profiles");
        
        String[] authorizedProjectProfileIds = StringUtils.split(StringUtils.defaultString(Config.getInstance().getValue("workspaces.profile.list")), ",");
        
        List<Profile> authorizedProjectProfile = Arrays.stream(authorizedProjectProfileIds)
              .map(_rightProfilesDao::getProfile)
              .filter(Objects::nonNull)
              .collect(Collectors.toList());
        
        for (Profile profile : authorizedProjectProfile)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", profile.getId());
            XMLUtils.startElement(contentHandler, "profile", attrs);
            XMLUtils.createElement(contentHandler, "label", profile.getLabel());
            XMLUtils.endElement(contentHandler, "profile");
        }
        
        XMLUtils.endElement(contentHandler, "profiles");
    }

}
